<?php
// File: includes/db-custom-urls.php
if ( ! defined( 'ABSPATH' ) ) { exit; }

global $wpdb;

/**
 * Table name helper.
 */
function ysmap_custom_urls_table_name() {
	global $wpdb;
	return $wpdb->prefix . 'sitemap_custom_urls';
}

/**
 * Ensure the custom table exists (safe to call repeatedly).
 */
function ysmap_ensure_custom_urls_table() {
	global $wpdb;
	$table           = ysmap_custom_urls_table_name();
	$charset_collate = $wpdb->get_charset_collate();

	// Does it already exist?
	$like   = $wpdb->esc_like( $table );
	$exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) );
	if ( $exists === $table ) {
		return;
	}

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	$sql = "CREATE TABLE `$table` (
		`id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
		`url` TEXT NOT NULL,
		`title` TEXT NOT NULL,
		`last_modified` DATE NOT NULL,
		`created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
		PRIMARY KEY (`id`),
		KEY `last_modified` (`last_modified`)
	) $charset_collate;";
	dbDelta( $sql );
}
add_action( 'admin_init', 'ysmap_ensure_custom_urls_table' );

/**
 * Cache helpers.
 */
function ysmap_custom_urls_cache_key() { return 'ysmap_custom_urls_all_v1'; }
function ysmap_custom_urls_cache_group() { return 'ysmap'; }
function ysmap_custom_urls_cache_bust() {
	wp_cache_delete( ysmap_custom_urls_cache_key(), ysmap_custom_urls_cache_group() );
	delete_transient( ysmap_custom_urls_cache_key() );
}

/**
 * Insert one custom URL row.
 * Accepts an associative array: [ 'url' => '', 'title' => '', 'last_modified' => 'YYYY-MM-DD' ].
 */
function ysmap_insert_custom_url( array $row ) {
	global $wpdb;
	$table = ysmap_custom_urls_table_name();

	$url           = isset( $row['url'] ) ? esc_url_raw( (string) $row['url'] ) : '';
	$title         = isset( $row['title'] ) ? sanitize_text_field( (string) $row['title'] ) : '';
	$last_modified = isset( $row['last_modified'] ) ? sanitize_text_field( (string) $row['last_modified'] ) : gmdate( 'Y-m-d' );

	if ( '' === $url ) {
		return false;
	}

	$ok = $wpdb->insert(
		$table,
		array(
			'url'           => $url,
			'title'         => $title,
			'last_modified' => $last_modified,
		),
		array( '%s', '%s', '%s' )
	);

	if ( false !== $ok ) {
		ysmap_custom_urls_cache_bust();
	}
	return $ok;
}

/**
 * Update one row by ID.
 * Pass fields in $row just like insert().
 */
function ysmap_update_custom_url( $id, array $row ) {
	global $wpdb;
	$table = ysmap_custom_urls_table_name();

	$data = array();
	$fmt  = array();

	if ( isset( $row['title'] ) ) {
		$data['title'] = sanitize_text_field( (string) $row['title'] );
		$fmt[] = '%s';
	}
	if ( isset( $row['url'] ) ) {
		$data['url'] = esc_url_raw( (string) $row['url'] );
		$fmt[] = '%s';
	}
	if ( isset( $row['last_modified'] ) ) {
		$data['last_modified'] = sanitize_text_field( (string) $row['last_modified'] );
		$fmt[] = '%s';
	}

	if ( empty( $data ) ) {
		return false;
	}

	$ok = $wpdb->update(
		$table,
		$data,
		array( 'id' => (int) $id ),
		$fmt,
		array( '%d' )
	);

	if ( false !== $ok ) {
		ysmap_custom_urls_cache_bust();
	}
	return $ok;
}

/**
 * Delete one row by ID.
 */
function ysmap_delete_custom_url( $id ) {
	global $wpdb;
	$table = ysmap_custom_urls_table_name();

	$ok = $wpdb->delete( $table, array( 'id' => (int) $id ), array( '%d' ) );
	if ( false !== $ok ) {
		ysmap_custom_urls_cache_bust();
	}
	return $ok;
}

/**
 * Get all custom URLs (cached).
 */
function ysmap_get_custom_urls() {
	global $wpdb;
	$table     = ysmap_custom_urls_table_name();
	$cache_key = ysmap_custom_urls_cache_key();
	$group     = ysmap_custom_urls_cache_group();

	// Object cache first
	$results = wp_cache_get( $cache_key, $group );
	if ( false !== $results ) {
		return $results;
	}
	// Transient fallback
	$results = get_transient( $cache_key );
	if ( false !== $results ) {
		wp_cache_set( $cache_key, $results, $group, 5 * MINUTE_IN_SECONDS );
		return $results;
	}

	// Verify table exists
	$like   = $wpdb->esc_like( $table );
	$exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) );
	if ( $exists !== $table ) {
		$results = array();
	} else {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
		$results = $wpdb->get_results( "SELECT * FROM `{$table}` ORDER BY `created_at` DESC" );
	}

	// Cache briefly
	wp_cache_set( $cache_key, $results, $group, 5 * MINUTE_IN_SECONDS );
	set_transient( $cache_key, $results, 5 * MINUTE_IN_SECONDS );

	return $results;
}
