<?php
/**
 * CSV Export for Sitemap Manager Pro
 *
 * Outputs a CSV of published posts plus custom URLs table (if present).
 * Expects a form posting to admin-post.php with:
 *   - hidden input: <input type="hidden" name="action" value="ysmap_export_csv">
 *   - nonce field  : wp_nonce_field( 'ysmap_export_csv', 'ysmap_export_csv_nonce' )
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Determine if a post is set to noindex under common SEO plugins.
 */
if ( ! function_exists( 'ysmap_is_noindex' ) ) {
    function ysmap_is_noindex( $post_id ) {
        // Sitewide "Discourage search engines"
        if ( get_option( 'blog_public' ) === '0' ) {
            return true;
        }

        $active = function_exists( 'ysmap_detect_active_seo_plugin' )
            ? ysmap_detect_active_seo_plugin()
            : 'none';

        switch ( $active ) {
            case 'yoast':
                return get_post_meta( $post_id, '_yoast_wpseo_meta-robots-noindex', true ) === '1';

            case 'rank_math':
                $rm = get_post_meta( $post_id, 'rank_math_robots', true );
                return ( is_string( $rm ) && stripos( $rm, 'noindex' ) !== false )
                    || ( is_array( $rm ) && in_array( 'noindex', $rm, true ) );

            case 'aioseo':
                $direct = get_post_meta( $post_id, '_aioseo_noindex', true );
                if ( $direct === '1' || $direct === 1 ) {
                    return true;
                }
                $robots = get_post_meta( $post_id, 'aioseo_robots_meta', true );
                if ( empty( $robots ) ) {
                    $robots = get_post_meta( $post_id, '_aioseo_robots_meta', true );
                }
                return ( is_string( $robots ) && stripos( $robots, 'noindex' ) !== false )
                    || ( is_array( $robots ) && ( in_array( 'noindex', $robots, true ) || ( ! empty( $robots['noindex'] ) ) ) );

            default:
                return false;
        }
    }
}

/**
 * Build a single CSV line from an array of fields.
 * RFC 4180 style: quote fields, escape quotes, normalize newlines.
 */
if ( ! function_exists( 'ysmap_csv_line' ) ) {
    function ysmap_csv_line( array $fields ) {
        foreach ( $fields as &$f ) {
            $f = (string) $f;
            $f = str_replace( array( "\r\n", "\r", "\n" ), ' ', $f ); // normalize newlines
            $f = str_replace( '"', '""', $f );                         // escape quotes
            $f = '"' . $f . '"';
        }
        return implode( ',', $fields ) . "\r\n";
    }
}

/**
 * Handle CSV export.
 */
if ( ! function_exists( 'ysmap_handle_export_csv' ) ) {
    function ysmap_handle_export_csv() {
        // Permission check (change to 'edit_others_posts' if you want editors to export)
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Unauthorized request', 'sitemap-manager-pro' ) );
        }

        // Nonce check (reads from $_REQUEST/$_POST internally)
        check_admin_referer( 'ysmap_export_csv', 'ysmap_export_csv_nonce' );

        $excluded_ids = get_option( 'ysmap_excluded_post_ids', array() );
        $post_types   = get_post_types( array( 'public' => true ), 'names' );

        $post_ids = get_posts( array(
            'post_type'      => $post_types,
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'orderby'        => 'title',
            'order'          => 'ASC',
            'no_found_rows'  => true,
        ) );

        if ( ! is_array( $post_ids ) ) {
            $post_ids = array();
        }

        $rows = array();

        // Core posts/pages/etc.
        foreach ( $post_ids as $post_id ) {
            $rows[] = array(
                get_the_title( $post_id ),
                get_the_date( 'Y-m-d', $post_id ),
                get_post_type( $post_id ),
                get_permalink( $post_id ),
                in_array( $post_id, $excluded_ids, true ) ? 'Yes' : 'No',
                ( function_exists( 'ysmap_is_noindex' ) && ysmap_is_noindex( $post_id ) ) ? 'Yes' : '',
            );
        }

        // Custom URLs table (if present)
        // Custom URLs table (if present), fetched via cached helper to avoid direct DB calls here.
        $custom_rows = array();
        if ( function_exists( 'ysmap_get_custom_urls' ) ) {
            $custom_rows = ysmap_get_custom_urls(); // returns cached results
        }
        
        // Map fields defensively (helper may return objects; older code used ARRAY_A).
        foreach ( (array) $custom_rows as $row ) {
            $is_obj = is_object( $row );
            $label  = $is_obj ? ( $row->label ?? $row->title ?? '' ) : ( $row['label'] ?? $row['title'] ?? '' );
            $url    = $is_obj ? ( $row->url   ?? '' )               : ( $row['url']   ?? '' );
            $date   = $is_obj
                ? ( $row->added_date ?? $row->last_modified ?? $row->created_at ?? '' )
                : ( $row['added_date'] ?? $row['last_modified'] ?? $row['created_at'] ?? '' );
            $active = $is_obj ? ( $row->active ?? null ) : ( $row['active'] ?? null );
        
            $rows[] = array(
                $label !== '' ? $label : '(No label)',
                $date,
                'Custom',
                $url,
                isset( $active ) ? ( ( (int) $active === 1 ) ? 'No' : 'Yes' ) : '',
                '',
            );
        }


        // Clean any stray buffers and send headers
        while ( ob_get_level() ) {
            @ob_end_clean(); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
        }

        nocache_headers();
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename="sitemap-export.csv"' );
        header( 'X-Content-Type-Options: nosniff' );

        // Build CSV
        $csv  = ysmap_csv_line( array( 'Title', 'Date', 'Post Type', 'URL', 'Excluded', 'Noindex' ) );
        foreach ( $rows as $r ) {
            $csv .= ysmap_csv_line( $r );
        }

        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- sending CSV, not HTML
        echo $csv;
        exit;
    }
}

// Hook the handler for logged-in users
add_action( 'admin_post_ysmap_export_csv', 'ysmap_handle_export_csv' );

// (Optional) If you ever want to allow non-logged-in requests (usually not),
// you could also hook: add_action( 'admin_post_nopriv_ysmap_export_csv', 'ysmap_handle_export_csv' );
