<?php

if ( ! defined( 'ABSPATH' ) ) exit;


define('SITEMAP_MANAGER_VERSION', '1.0.6'); // Plugin version
define('SITEMAP_MANAGER_NAME', 'Sitemap Manager Pro');

if (!defined('YSMAP_MAX_FREE')) {
    define('YSMAP_MAX_FREE', 100);
}

// Insert one custom URL row.
function ysmap_insert_custom_url( $url, $title, $last_modified ) {
    global $wpdb;
    $table = $wpdb->prefix . 'sitemap_custom_urls';

    $inserted = $wpdb->insert(
        $table,
        array(
            'url'           => $url,
            'title'         => $title,
            'last_modified' => $last_modified,
        ),
        array( '%s', '%s', '%s' )
    );

    if ( false !== $inserted ) {
        ysmap_custom_urls_cache_bust();
    }
    return $inserted;
}

// Update one row by ID.
function ysmap_update_custom_url( $id, $title, $url, $last_modified ) {
    global $wpdb;
    $table = $wpdb->prefix . 'sitemap_custom_urls';

    $updated = $wpdb->update(
        $table,
        array(
            'title'         => $title,
            'url'           => $url,
            'last_modified' => $last_modified,
        ),
        array( 'id' => (int) $id ),
        array( '%s', '%s', '%s' ),
        array( '%d' )
    );

    if ( false !== $updated ) {
        ysmap_custom_urls_cache_bust();
    }
    return $updated;
}

// Delete one row by ID.
function ysmap_delete_custom_url( $id ) {
    global $wpdb;
    $table = $wpdb->prefix . 'sitemap_custom_urls';

    $deleted = $wpdb->delete(
        $table,
        array( 'id' => (int) $id ),
        array( '%d' )
    );

    if ( false !== $deleted ) {
        ysmap_custom_urls_cache_bust();
    }
    return $deleted;
}

// Cached fetch of custom URLs.
function ysmap_get_custom_urls() {
    global $wpdb;

    $table     = $wpdb->prefix . 'sitemap_custom_urls';
    $cache_key = 'ysmap_custom_urls_all_v2'; // bump key to avoid stale shape
    $group     = 'ysmap';

    // Try object cache first.
    $results = wp_cache_get( $cache_key, $group );
    if ( false !== $results ) {
        return $results;
    }

    // Fallback to transient.
    $results = get_transient( $cache_key );
    if ( false !== $results ) {
        wp_cache_set( $cache_key, $results, $group, 5 * MINUTE_IN_SECONDS );
        return $results;
    }

    // Verify table exists (exact match).
    $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );

    if ( $exists !== $table ) {
        $results = array();
    } else {
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- deterministic table identifier
        $sql = "
            SELECT
                id,
                url,
                title,
                label,
                COALESCE(NULLIF(label,''), title) AS display_title,
                last_modified,
                created_at
            FROM `{$table}`
            ORDER BY created_at DESC
        ";
        // Return OBJECTs (default) so $row->prop works in the view.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $results = $wpdb->get_results( $sql );
    }

    wp_cache_set( $cache_key, $results, $group, 5 * MINUTE_IN_SECONDS );
    set_transient( $cache_key, $results, 5 * MINUTE_IN_SECONDS );

    return $results;
}

function ysmap_normalize_host( $host ) {
    $host = is_string( $host ) ? strtolower( $host ) : '';
    $host = rtrim( $host, '.' );
    if ( substr( $host, 0, 4 ) === 'www.' ) {
        $host = substr( $host, 4 );
    }
    return $host;
}


// Call this after any insert/update/delete on the custom table.
function ysmap_custom_urls_cache_bust() {
    $cache_key = 'ysmap_custom_urls_all_v2';
    $group     = 'ysmap';
    wp_cache_delete( $cache_key, $group );
    delete_transient( $cache_key );
}



function ysmap_get_max_free_count() {
    $license = get_option('ysmap_license_key');
    $valid_until = get_option('ysmap_license_valid_until');

    if (!empty($license) && !empty($valid_until) && strtotime($valid_until) >= time()) {
        return PHP_INT_MAX; // License is still active
    }

    return YSMAP_MAX_FREE;
}

function ysmap_get_free_level_number() {
    return YSMAP_MAX_FREE;
}

function ysmap_check_license_expiration() {
    $valid_until = get_option('ysmap_license_valid_until');
    if ($valid_until) {
        $now = time();
        $expiration = strtotime($valid_until);

        if ($now > $expiration) {
            update_option('ysmap_license_status', 'expired');
        }
    }
}
add_action('init', 'ysmap_check_license_expiration');

function ysmap_get_maxfree_post_date() {
    $max_free = ysmap_get_max_free_count();

    $args = [
        'post_type'      => array_keys(get_post_types(['public' => true])),
        'post_status'    => 'publish',
        'posts_per_page' => 1,
        'orderby'        => 'date',
        'order'          => 'ASC',
        'offset'         => $max_free - 1,
        'fields'         => 'ids',
    ];

    $posts = get_posts($args);

    if (!empty($posts)) {
        $post = get_post($posts[0]);
        return $post ? $post->post_date : null;
    }

    return null;
}

function ysmap_get_limited_scope_ids($max_free = 500) {
    $all_public_post_types = array_keys(get_post_types(['public' => true]));

    return get_posts([
        'post_type'      => $all_public_post_types,
        'post_status'    => 'publish',
        'posts_per_page' => $max_free,
        'fields'         => 'ids',
        'orderby'        => 'date',
        'order'          => 'ASC',
    ]);
}

function ysmap_get_total_url_count() {
    $cached = get_transient('ysmap_cached_url_count');
    if ($cached !== false) return (int) $cached;

    $query = new WP_Query([
        'post_type'      => array_keys(get_post_types(['public' => true])),
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'fields'         => 'ids',
    ]);
    $count = $query->post_count;

    set_transient('ysmap_cached_url_count', $count, 3600);
    return $count;
}

function ysmap_get_active_seo_plugin() {
    if (defined('WPSEO_VERSION')) return 'yoast';
    if (defined('RANK_MATH_VERSION')) return 'rankmath';
    if (defined('AIOSEO_VERSION')) return 'aioseo';
    return 'native';
}

// Exclude posts from sitemap (Yoast)
add_filter('wpseo_exclude_from_sitemap_by_post_ids', function($excluded) {
    $custom_excluded = get_option('ysmap_excluded_post_ids', []);
    if (is_array($custom_excluded)) {
        $excluded = array_merge($excluded, $custom_excluded);
    }
    return array_unique($excluded);
});

// Exclude posts from sitemap (Rank Math)
add_filter('rank_math/sitemap/exclude_post_ids', function($excluded) {
    $custom_excluded = get_option('ysmap_excluded_post_ids', []);
    if (is_array($custom_excluded)) {
        $excluded = array_merge($excluded, $custom_excluded);
    }
    return array_unique($excluded);
});

// Exclude posts from sitemap (AIOSEO)
add_filter('aioseo_sitemap_excluded_post_ids', function($excluded) {
    $custom_excluded = get_option('ysmap_excluded_post_ids', []);
    if (is_array($custom_excluded)) {
        $excluded = array_merge($excluded, $custom_excluded);
    }
    return array_unique($excluded);
});

function ysmap_clear_seo_sitemap_cache() {
    // Yoast
    if (function_exists('wpseo_clear_sitemap_cache')) {
        wpseo_clear_sitemap_cache();
    }

    // Rank Math
    if (class_exists('\\RankMath\\Sitemap\\Cache') && method_exists('\\RankMath\\Sitemap\\Cache', 'clear')) {
        \RankMath\Sitemap\Cache::clear();

    }
}

// --- SEO detail helpers (safe to paste into includes/helpers.php) ---

// H1 extractor (DOMDocument with regex fallback); falls back to post title if none found
if ( ! function_exists('ysmap_get_h1_info') ) {
    function ysmap_get_h1_info( WP_Post $post ): array {
        $content = $post->post_content ?? '';

        // Build HTML we’ll parse: expand shortcodes but don’t let them mutate $GLOBALS['post']
        $old_post_global = $GLOBALS['post'] ?? null;
        $html = do_shortcode( $content );
        $GLOBALS['post'] = $old_post_global;

        $count = 0; 
        $first = '';

        if ( $html !== '' && class_exists('DOMDocument') ) {
            $dom = new DOMDocument();
            libxml_use_internal_errors(true);
            $loaded = $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html);
            libxml_clear_errors();
            if ( $loaded ) {
                $h1s = $dom->getElementsByTagName('h1');
                $count = $h1s->length ?? 0;
                if ( $count > 0 ) {
                    $first = trim( $h1s->item(0)->textContent ?? '' );
                }
            }
        }

        // Regex fallback if DOM failed / unavailable
        if ( $count === 0 && $html !== '' ) {
            if ( preg_match_all('/<h1\b[^>]*>(.*?)<\/h1>/is', $html, $m) ) {
                $count = count($m[0]);
                if ( $count > 0 ) {
                    $first = trim( wp_strip_all_tags( $m[1][0] ) );
                }
            }
        }

        // Theme-level fallback: use the post title as the page H1 if we still found nothing
        if ( $count === 0 ) {
            $title = get_the_title( $post );
            if ( is_string($title) && $title !== '' ) {
                return ['count'=>1, 'first'=>$title, 'status'=>'green', 'msg'=>'Using post title as theme H1.'];
            }
        }

        if ( $count === 1 ) {
            return ['count'=>1, 'first'=>$first, 'status'=>'green', 'msg'=>'1 H1 found.'];
        } elseif ( $count === 0 ) {
            return ['count'=>0, 'first'=>'', 'status'=>'red', 'msg'=>'No H1 found.'];
        }
        return ['count'=>$count, 'first'=>$first, 'status'=>'red', 'msg'=>"Found {$count} H1s. Consider one."];
    }
}


// Meta description: SEO plugin fields → excerpt → trimmed content
/**
 * META DESCRIPTION
 * - Resolves Yoast %%placeholders%% via wpseo_replace_vars
 * - Falls back to Rank Math / AIOSEO / excerpt
 * - Returns length cautions (short / good / long)
 */
function ysmap_get_meta_desc( int $post_id ): array {
    $post = get_post( $post_id );
    if ( ! $post ) {
        return ['text' => '', 'len' => 0, 'status' => 'red', 'msg' => 'Post not found'];
    }

    $desc = '';

    // 1) Yoast
    $yoast = get_post_meta( $post_id, '_yoast_wpseo_metadesc', true );
    if ( is_string($yoast) && $yoast !== '' ) {
        // If it contains %%vars%%, resolve them. If Yoast isn’t active, filter will no-op.
        if ( strpos( $yoast, '%%' ) !== false ) {
            $yoast_resolved = apply_filters( 'wpseo_replace_vars', $yoast, $post );
            // If the filter didn’t resolve, strip the raw tokens so we don’t show %%excerpt%%, etc.
            $desc = trim( $yoast_resolved ?: preg_replace('/%%[^%]+%%/', '', $yoast) );
        } else {
            $desc = trim( $yoast );
        }
    }

    // 2) Rank Math
    if ( $desc === '' ) {
        $rank = get_post_meta( $post_id, 'rank_math_description', true );
        if ( is_string($rank) ) {
            $desc = trim( $rank );
        }
    }

    // 3) All in One SEO
    if ( $desc === '' ) {
        $aio = get_post_meta( $post_id, '_aioseo_description', true );
        if ( is_string($aio) ) {
            $desc = trim( $aio );
        }
    }

    // 4) Fallback: build from content/excerpt (strip shortcodes, tags)
    if ( $desc === '' ) {
        $content  = get_post_field( 'post_content', $post_id );
        $content  = do_shortcode( $content );
        $text     = wp_strip_all_tags( $content );
        // Prefer excerpt if present.
        $excerpt  = has_excerpt( $post_id ) ? get_the_excerpt( $post_id ) : '';
        $source   = $excerpt !== '' ? $excerpt : $text;
        $desc     = wp_trim_words( $source, 30, '' ); // ~155–165 chars typical
        $desc     = trim( preg_replace('/\s+/', ' ', $desc) );
    }

    // Normalize whitespace and remove any lingering %%tokens%%
    $desc = trim( preg_replace('/\s+/', ' ', preg_replace('/%%[^%]+%%/', '', $desc) ) );

    $plain = wp_strip_all_tags( $desc );
    $len   = mb_strlen( $plain );

    // Length guidance
    // (roughly: <50 too short; 50–160 ideal; 161–300 long; >300 very long)
    if ( $len === 0 ) {
        return ['text' => '', 'len' => 0, 'status' => 'red', 'msg' => 'No meta description found'];
    } elseif ( $len < 50 ) {
        return ['text' => $plain, 'len' => $len, 'status' => 'yellow', 'msg' => 'Very short; consider adding more context'];
    } elseif ( $len <= 160 ) {
        return ['text' => $plain, 'len' => $len, 'status' => 'green', 'msg' => 'Good length'];
    } elseif ( $len <= 300 ) {
        return ['text' => $plain, 'len' => $len, 'status' => 'yellow', 'msg' => 'Long; mobile truncation likely'];
    } else {
        return ['text' => $plain, 'len' => $len, 'status' => 'red', 'msg' => 'Too long; truncate for clarity'];
    }
}

/**
 * H1 INFO
 * - Counts H1s from rendered content (after the_content filters)
 * - Also looks at raw Gutenberg blocks in case filters don’t render
 * - Returns count + first H1 text + status/caution
 */
function ysmap_get_h1_info( WP_Post $post ): array {
    $post_id = $post->ID;

    // Render content through filters so Gutenberg headings become <h1> tags.
    $content_rendered = apply_filters( 'the_content', get_post_field( 'post_content', $post_id ) );
    $raw_content      = get_post_field( 'post_content', $post_id );

    $count = 0;
    $first = '';

    // 1) Count real <h1> tags in rendered HTML
    if ( is_string( $content_rendered ) && $content_rendered !== '' ) {
        if ( preg_match_all( '/<h1\b[^>]*>(.*?)<\/h1>/is', $content_rendered, $m ) ) {
            $count += count( $m[0] );
            if ( $first === '' && ! empty( $m[1][0] ) ) {
                $first = trim( wp_strip_all_tags( $m[1][0] ) );
            }
        }
    }

    // 2) Safety net: detect Gutenberg heading blocks with "level":1 that for some reason didn’t render
    if ( $raw_content ) {
        if ( preg_match_all( '/<!--\s*wp:heading\s*({[^}]*"level"\s*:\s*1[^}]*)?\s*-->.*?<!--\s*\/wp:heading\s*-->/is', $raw_content, $gm ) ) {
            // If rendered didn’t produce <h1>, count these as H1s
            if ( $count === 0 ) {
                $count = count( $gm[0] );
                // Try to extract first block’s text
                if ( preg_match( '/<!--\s*wp:heading.*?-->(.*?)<!--\s*\/wp:heading\s*-->/is', $raw_content, $firstBlock ) ) {
                    $first = trim( wp_strip_all_tags( $firstBlock[1] ) );
                }
            }
        }
    }

    // Status & message
    if ( $count === 0 ) {
        return ['count' => 0, 'first' => '', 'status' => 'red',    'msg' => 'No H1 found'];
    } elseif ( $count === 1 ) {
        return ['count' => 1, 'first' => $first, 'status' => 'green',  'msg' => 'Exactly one H1'];
    } else {
        return ['count' => $count, 'first' => $first, 'status' => 'yellow', 'msg' => 'Multiple H1s found'];
    }
}

/**
 * (Optional) Canonical helper, unchanged here—only included for completeness.
 * Keep your existing implementation if it’s already working.
 */
function ysmap_get_canonical_info( int $post_id ): array {
    $canonical = '';
    // Yoast
    $yoast_canon = get_post_meta( $post_id, '_yoast_wpseo_canonical', true );
    if ( $yoast_canon ) $canonical = $yoast_canon;

    // Rank Math
    if ( $canonical === '' ) {
        $rank_canon = get_post_meta( $post_id, 'rank_math_canonical_url', true );
        if ( $rank_canon ) $canonical = $rank_canon;
    }

    // Fallback to permalink
    if ( $canonical === '' ) $canonical = get_permalink( $post_id );

    $page_url = get_permalink( $post_id );
    $matches  = untrailingslashit( $canonical ) === untrailingslashit( $page_url );

    return [
        'url'    => esc_url_raw( $canonical ),
        'status' => $matches ? 'green' : 'yellow',
        'msg'    => $matches ? 'Canonical matches page URL.' : 'Canonical differs from page URL.',
    ];
}

// Noindex detector: Yoast / Rank Math / AIOSEO / SEOPress
if ( ! function_exists('ysmap_is_noindex') ) {
    function ysmap_is_noindex( $post_id ) {
        if ( ! $post_id ) return false;
        // Yoast
        $yoast = get_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', true);
        if ($yoast === '1') return true;

        // Rank Math
        $rank = get_post_meta($post_id, 'rank_math_robots', true);
        if ( is_string($rank) && stripos($rank, 'noindex') !== false ) return true;

        // AIOSEO
        $aioseo_noindex = get_post_meta($post_id, '_aioseo_noindex', true);
        if ($aioseo_noindex === '1') return true;
        $aioseo_robots = get_post_meta($post_id, 'aioseo_robots_meta', true);
        if ( is_string($aioseo_robots) && stripos($aioseo_robots, 'noindex') !== false ) return true;
        $aioseo_robots_alt = get_post_meta($post_id, '_aioseo_robots_meta', true);
        if ( is_string($aioseo_robots_alt) && stripos($aioseo_robots_alt, 'noindex') !== false ) return true;

        // SEOPress (stores "yes" when noindex)
        $seopress = get_post_meta($post_id, '_seopress_robots_index', true);
        if ($seopress === 'yes') return true;

        return false;
    }
}

// Index/noindex filter pusher for WP_Query meta_query
if ( ! function_exists('ysmap_apply_noindex_meta_query') ) {
    function ysmap_apply_noindex_meta_query( array &$args, $mode /* 'noindex'|'index' */ ) {
        $active = function_exists('ysmap_get_active_seo_plugin')
            ? ysmap_get_active_seo_plugin()
            : 'native';

        $mq = null;

        if ($active === 'yoast') {
            $mq = ($mode === 'noindex')
                ? [ [ 'key'=>'_yoast_wpseo_meta-robots-noindex', 'value'=>'1', 'compare'=>'=' ] ]
                : [
                    'relation'=>'OR',
                    [ 'key'=>'_yoast_wpseo_meta-robots-noindex', 'compare'=>'NOT EXISTS' ],
                    [ 'key'=>'_yoast_wpseo_meta-robots-noindex', 'value'=>'1', 'compare'=>'!=' ],
                ];
        } elseif ($active === 'rankmath' || $active === 'rank_math') {
            $mq = ($mode === 'noindex')
                ? [ [ 'key'=>'rank_math_robots', 'value'=>'noindex', 'compare'=>'LIKE' ] ]
                : [
                    'relation'=>'OR',
                    [ 'key'=>'rank_math_robots', 'compare'=>'NOT EXISTS' ],
                    [ 'key'=>'rank_math_robots', 'value'=>'noindex', 'compare'=>'NOT LIKE' ],
                ];
        } elseif ($active === 'aioseo') {
            $mq = ($mode === 'noindex')
                ? [
                    'relation'=>'OR',
                    [ 'key'=>'_aioseo_noindex', 'value'=>'1', 'compare'=>'=' ],
                    [ 'key'=>'aioseo_robots_meta',  'value'=>'noindex', 'compare'=>'LIKE' ],
                    [ 'key'=>'_aioseo_robots_meta', 'value'=>'noindex', 'compare'=>'LIKE' ],
                ]
                : [
                    'relation'=>'AND',
                    [
                        'relation'=>'OR',
                        [ 'key'=>'_aioseo_noindex', 'compare'=>'NOT EXISTS' ],
                        [ 'key'=>'_aioseo_noindex', 'value'=>'1', 'compare'=>'!=' ],
                    ],
                    [
                        'relation'=>'OR',
                        [ 'key'=>'aioseo_robots_meta',  'compare'=>'NOT EXISTS' ],
                        [ 'key'=>'aioseo_robots_meta',  'value'=>'noindex', 'compare'=>'NOT LIKE' ],
                    ],
                    [
                        'relation'=>'OR',
                        [ 'key'=>'_aioseo_robots_meta', 'compare'=>'NOT EXISTS' ],
                        [ 'key'=>'_aioseo_robots_meta', 'value'=>'noindex', 'compare'=>'NOT LIKE' ],
                    ],
                ];
        }

        if ($mq) {
            if (!isset($args['meta_query'])) $args['meta_query'] = [];
            if (!isset($args['meta_query']['relation'])) $args['meta_query']['relation'] = 'AND';
            $args['meta_query'][] = $mq;
        }
    }
}

// Directory helpers used by sitemap.php (filter dropdown and query narrowing)
// In includes/helpers.php (or wherever ysmap_get_directory_filters() lives)
if ( ! function_exists('ysmap_get_directory_filters') ) {
    function ysmap_get_directory_filters(): array {
        $cache_key = 'ysmap_dir_filters_v2';
        $cached    = get_transient($cache_key);
        if ( is_array($cached) ) {
            return $cached;
        }

        // Normalize site host (no www)
        $home_host = wp_parse_url(home_url(), PHP_URL_HOST) ?: '';
        $home_host = strtolower(preg_replace('/^www\./i', '', $home_host));

        $paths = [];

        // 1) Gather published posts/pages/CPT permalinks
        $post_types = get_post_types(['public' => true], 'names');
        unset($post_types['attachment']); // optional
        $q = new WP_Query([
            'post_type'           => array_values($post_types),
            'post_status'         => 'publish',
            'posts_per_page'      => -1,
            'fields'              => 'ids',
            'no_found_rows'       => true,
            'ignore_sticky_posts' => true,
        ]);
        foreach ( (array) $q->posts as $pid ) {
            $url  = get_permalink($pid);
            $host = $url ? wp_parse_url($url, PHP_URL_HOST) : '';
            $host = strtolower(preg_replace('/^www\./i', '', (string) $host));
            if ($host && $host !== $home_host) { continue; }
            $path = $url ? (wp_parse_url($url, PHP_URL_PATH) ?: '/') : '/';
            $paths[] = $path;
        }

        // 2) Public taxonomy term links (optional but nice)
        $taxes = get_taxonomies(['public' => true], 'names');
        if ($taxes) {
            $term_ids = get_terms(['taxonomy' => array_values($taxes), 'hide_empty' => true, 'fields' => 'ids']);
            if ( ! is_wp_error($term_ids) ) {
                foreach ( (array) $term_ids as $tid ) {
                    $url  = get_term_link((int) $tid);
                    if ( is_wp_error($url) ) { continue; }
                    $host = $url ? wp_parse_url($url, PHP_URL_HOST) : '';
                    $host = strtolower(preg_replace('/^www\./i', '', (string) $host));
                    if ($host && $host !== $home_host) { continue; }
                    $path = $url ? (wp_parse_url($url, PHP_URL_PATH) ?: '/') : '/';
                    $paths[] = $path;
                }
            }
        }

        // 3) Custom URLs from Pro table (if present)
        global $wpdb;
        $table = $wpdb->prefix . 'sitemap_custom_urls';
        $exists = $wpdb->get_var( $wpdb->prepare("SHOW TABLES LIKE %s", $table) );
        if ( $exists === $table ) {
            $urls = (array) $wpdb->get_col("SELECT url FROM `$table` WHERE url <> ''");
            foreach ( $urls as $url ) {
                if ( ! is_string($url) ) { continue; }
                $path = '/';
                if ( str_starts_with($url, '/') ) {
                    $path = $url;
                } else {
                    $host = wp_parse_url($url, PHP_URL_HOST);
                    $host = strtolower(preg_replace('/^www\./i', '', (string) $host));
                    if ($host && $host !== $home_host) { continue; }
                    $path = wp_parse_url($url, PHP_URL_PATH) ?: '/';
                }
                $paths[] = $path;
            }
        }

        // Build directory list from first path segment, but ONLY if there is deeper depth
        $segments = [];
        foreach ( $paths as $path ) {
            $norm  = '/' . trim($path, '/') . '/';
            $parts = array_values(array_filter(explode('/', trim($norm, '/'))));
            if ( count($parts) < 2 ) { // depth < 2 → just a top-level page; skip
                continue;
            }
            $first = $parts[0];
            if ($first === '') { continue; }
            // Optional: skip numeric year-like segments (e.g., /2025/)
            if ( ctype_digit($first) ) { continue; }
            $segments[$first] = true;
        }

        $dirs = array_map(fn($seg) => '/' . $seg . '/', array_keys($segments));
        sort($dirs, SORT_NATURAL | SORT_FLAG_CASE);

        set_transient($cache_key, $dirs, DAY_IN_SECONDS);
        return $dirs;
    }
}




if ( ! function_exists('ysmap_get_posts_by_directory') ) {
    /**
     * Return post IDs that live under a given TOP-LEVEL directory (e.g., "/blog/").
     * - $dir may be "blog" or "/blog/" (normalized internally).
     * - If $dir === "" (empty), returns ALL published public post IDs.
     * - Safe for sites installed in a subdirectory (e.g. /wp/).
     */
    function ysmap_get_posts_by_directory( string $dir = '' ): array {
        $dir = trim($dir);
        $want_all = ($dir === '');

        // Normalize directory to bare segment name (no slashes)
        $needle = $want_all ? '' : trim($dir, '/');

        // Cache key
        $ck = 'ysmap_dir_ids_top_' . ($want_all ? 'ALL' : md5($needle));
        $cached = get_transient($ck);
        if ($cached !== false) return $cached;

        // Public post types only
        $public_types = array_keys( get_post_types(['public' => true]) );

        $q = new WP_Query([
            'post_type'      => $public_types,
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'no_found_rows'  => true,
        ]);

        $ids = [];
        if ( $q->have_posts() ) {
            // Determine base path segments for home_url (handles subdirectory installs)
            $home_path = wp_parse_url( home_url( '/' ), PHP_URL_PATH );
            $base_segments = $home_path ? array_values(array_filter(explode('/', trim($home_path, '/')))) : [];
            $base_len = count($base_segments);

            foreach ( $q->posts as $pid ) {
                $path = wp_parse_url( get_permalink($pid), PHP_URL_PATH );
                if (!is_string($path) || $path === '') continue;

                $segments = array_values(array_filter(explode('/', trim($path, '/'))));
                if (empty($segments)) continue;

                // Strip site base segments (e.g., '/wp/') to get site-relative path
                if ($base_len > 0) {
                    $segments = array_slice($segments, $base_len);
                    if (empty($segments)) continue;
                }

                if ( $want_all ) {
                    $ids[] = (int) $pid;
                    continue;
                }

                // Need at least two segments to be in a directory: /dir/slug
                if (count($segments) < 2) continue;

                // Top-level directory = the first segment after base
                if ($segments[0] === $needle) {
                    $ids[] = (int) $pid;
                }
            }
        }
        wp_reset_postdata();

        // Cache 15 minutes
        set_transient($ck, $ids, 15 * MINUTE_IN_SECONDS);
        return $ids;
    }
}

