<?php
// File: plk/plk-loader.php

if ( ! defined( 'ABSPATH' ) ) { exit; }

// Simple local/state check used by the UI and gating logic
if ( ! function_exists('plk_is_license_valid') ) {
    function plk_is_license_valid(): bool {
	   // Handy bypass while developing
	   if ( defined('PLK_DEV_MODE') && PLK_DEV_MODE ) {
		  return true;
	   }
	   $status  = get_option('plk_license_status');
	   $expires = strtotime( (string) get_option('plk_license_valid_until') );
	   return ($status === 'valid') && $expires && ($expires > time());
    }
}

function plk_get_site_domain(): string {
    $host = wp_parse_url( home_url(), PHP_URL_HOST );
    if ( ! is_string( $host ) || $host === '' ) {
	   $host = wp_parse_url( site_url(), PHP_URL_HOST );
	   if ( ! is_string( $host ) || $host === '' ) return '';
    }
    $host = strtolower( $host );
    return preg_replace( '/^www\./i', '', $host );
}

/**
 * Registers plugin with PLK and optionally boots the updater
 * @param array $args ['slug','name','license_check_url','version','helper_text','plugin_file','update_check_url','license_option','param_map']
 */
function plk_register_plugin( $args ) {
    $defaults = array(
	   'slug'              => '',
	   'name'              => 'This plugin',
	   'license_check_url' => '',
	   'version'           => '',
	   'helper_text'       => '',
	   // Updater-related (optional)
	   'plugin_file'       => '',
	   'update_check_url'  => '',
	   'license_option'    => 'plk_license_key',
	   'param_map'         => array(
		  'license' => 'license',
		  'site'    => 'site',
		  'slug'    => 'slug',
		  'version' => 'version',
	   ),
    );
    $plugin = array_merge( $defaults, (array) $args );

    // Store meta (NOTE: single-plugin assumption; if you need multi-plugin, key by slug)
    update_option( 'plk_plugin_meta', $plugin );

    add_action( 'admin_footer', 'plk_render_license_modal' );
    add_action( 'admin_enqueue_scripts', 'plk_enqueue_plk_assets' );

    // Boot updater if configured
    if (
	   ! empty( $plugin['update_check_url'] ) &&
	   ! empty( $plugin['plugin_file'] ) &&
	   ! empty( $plugin['slug'] ) &&
	   ! empty( $plugin['version'] ) &&
	   class_exists( 'PLK_Updater' )
    ) {
	   new PLK_Updater( $plugin );
    }
}

/**
 * Verify a license key with the remote PLK endpoint.
 *
 * Returns array( 'ok' => bool, 'data' => array|WP_Error )
 * On success, 'data' is a normalized array with keys:
 *   status, activated_on, renewal_date, name, email, license_id, domain
 */
function plk_verify_license_remote( $key ) {
	$key = sanitize_text_field( (string) $key );
 
	$meta = get_option( 'plk_plugin_meta', array() );
	$url  = isset( $meta['license_check_url'] ) ? esc_url_raw( $meta['license_check_url'] ) : '';
 
	if ( empty( $url ) ) {
	    return array( 'ok' => false, 'data' => new WP_Error( 'plk_no_endpoint', 'License check URL not configured.' ) );
	}
 
	// Apex (no www)
	$domain = plk_get_site_domain();
 
	// DO NOT pre-encode; add_query_arg() will handle it.
	$req_url = add_query_arg(
	    array(
		   'key'    => $key,
		   'domain' => $domain,
	    ),
	    $url
	);
 
	$response = wp_remote_get( $req_url, array( 'timeout' => 10 ) );
	if ( is_wp_error( $response ) ) {
	    return array( 'ok' => false, 'data' => $response );
	}
	if ( 200 !== wp_remote_retrieve_response_code( $response ) ) {
	    return array( 'ok' => false, 'data' => new WP_Error( 'plk_http', 'Unexpected response code.' ) );
	}
 
	$body = json_decode( (string) wp_remote_retrieve_body( $response ), true );
	if ( ! is_array( $body ) ) {
	    return array( 'ok' => false, 'data' => new WP_Error( 'plk_bad_json', 'Invalid JSON from license server.' ) );
	}
 
	$normalized = array(
	    'status'       => isset( $body['status'] ) ? (string) $body['status'] : '',
	    'activated_on' => isset( $body['activated_on'] ) ? sanitize_text_field( $body['activated_on'] ) : '',
	    'renewal_date' => isset( $body['renewal_date'] ) ? sanitize_text_field( $body['renewal_date'] ) : ( isset( $body['expires'] ) ? sanitize_text_field( $body['expires'] ) : '' ),
	    'name'         => isset( $body['name'] ) ? sanitize_text_field( $body['name'] ) : '',
	    'email'        => isset( $body['email'] ) ? sanitize_email( $body['email'] ) : '',
	    'license_id'   => isset( $body['license_id'] ) ? sanitize_text_field( $body['license_id'] ) : '',
	    'domain'       => isset( $body['domain'] ) ? sanitize_text_field( $body['domain'] ) : $domain,
	);
 
	return array( 'ok' => true, 'data' => $normalized );
 }


/**
 * Persist license data options.
 */
function plk_save_license_data( $key, array $data ) {
	// Always store in PLK’s defaults…
	update_option( 'plk_license_key', sanitize_text_field( (string) $key ) );
	// …and also in the plugin’s configured option (so the updater sees it).
	$meta = get_option( 'plk_plugin_meta', array() );
	$opt  = isset( $meta['license_option'] ) ? sanitize_key( $meta['license_option'] ) : '';
	if ( $opt ) {
	    update_option( $opt, sanitize_text_field( (string) $key ) );
	}
 
	update_option( 'plk_license_status', (string) ( $data['status'] ?? '' ) );
	update_option( 'plk_license_active_date', sanitize_text_field( (string) ( $data['activated_on'] ?? '' ) ) );
	update_option( 'plk_license_valid_until', sanitize_text_field( (string) ( $data['renewal_date'] ?? '' ) ) );
	update_option( 'plk_license_name', sanitize_text_field( (string) ( $data['name'] ?? '' ) ) );
	update_option( 'plk_license_email', sanitize_email( (string) ( $data['email'] ?? '' ) ) );
	update_option( 'plk_license_id', sanitize_text_field( (string) ( $data['license_id'] ?? '' ) ) );
	update_option( 'plk_domain', sanitize_text_field( (string) ( $data['domain'] ?? '' ) ) );
 }


/**
 * Clear license data options.
 */
function plk_clear_license_data() {
	delete_option( 'plk_license_key' );
	delete_option( 'plk_license_status' );
	delete_option( 'plk_license_active_date' );
	delete_option( 'plk_license_valid_until' );
	delete_option( 'plk_license_name' );
	delete_option( 'plk_license_email' );
	delete_option( 'plk_license_id' );
	delete_option( 'plk_domain' );
}

/**
 * Enqueue JS & CSS and localize license/meta to JS
 */
function plk_enqueue_plk_assets() {
	$base_url = plugin_dir_url( __FILE__ );
	$version  = time(); // or your plugin version constant

	wp_enqueue_style( 'plk-modal-css', $base_url . 'plk-modal.css', array(), $version );
	wp_enqueue_script( 'plk-modal-js', $base_url . 'plk-modal.js', array( 'jquery' ), $version, true );

	$plugin_meta = get_option( 'plk_plugin_meta', array() );

	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only
	$current_page = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : '';

	wp_localize_script(
		'plk-modal-js',
		'plkData',
		array(
			'license_valid'    => (bool) plk_is_license_valid(),
			'current_page'     => $current_page,
			'allowed_pages'    => defined( 'PLK_ALLOW_UNLICENSED_PAGES' ) ? (array) PLK_ALLOW_UNLICENSED_PAGES : array(),
			'nonce'            => wp_create_nonce( 'plk_clear_license' ),
			'activation_nonce' => wp_create_nonce( 'plk_activate_license' ),
			'plugin_slug'      => isset( $plugin_meta['slug'] ) ? sanitize_key( $plugin_meta['slug'] ) : '',
		)
	);

	$license_data = array(
		'key'          => get_option( 'plk_license_key' ),
		'status'       => get_option( 'plk_license_status' ),
		'activated_on' => get_option( 'plk_license_active_date' ),
		'valid_until'  => get_option( 'plk_license_valid_until' ),
		'name'         => get_option( 'plk_license_name' ),
		'email'        => get_option( 'plk_license_email' ),
		'domain'       => get_option( 'plk_domain' ),
		'id'           => get_option( 'plk_license_id' ),
	);
	wp_localize_script( 'plk-modal-js', 'plkLicenseData', $license_data );
}

/**
 * Render the license modal markup
 */
function plk_render_license_modal() {
	if ( ! defined( 'ABSPATH' ) ) { exit; }

	$plugin          = get_option( 'plk_plugin_meta' );
	$plugin_name     = isset( $plugin['name'] ) ? esc_html( $plugin['name'] ) : 'This Plugin';
	$plugin_slug     = isset( $plugin['slug'] ) ? sanitize_key( $plugin['slug'] ) : '';
	$helper_text_raw = isset( $plugin['helper_text'] ) ? $plugin['helper_text'] : '';
	$helper_text     = $helper_text_raw ? wp_kses_post( $helper_text_raw ) : '';
	$activation_nonce = wp_create_nonce( 'plk_activate_license' );

	

	// Overlay
	echo '<div id="plk-modal-overlay" class="plk-hidden" aria-hidden="true"></div>';

	// BLOCKED modal
	echo '<div id="plk-modal-blocked" class="plk-modal-box plk-hidden" role="dialog" aria-modal="true" aria-labelledby="plk-blocked-title">';
	echo '<h2 id="plk-blocked-title">License Required</h2>';

	if ( $helper_text ) {
		printf( '<div class="plk-helper-text">%s</div>', $helper_text );
	}

	echo '<form method="post" action="">';
	echo '<input type="text" name="plk_license_key" placeholder="Enter your license key" required>';
	echo '<input type="hidden" name="_wpnonce" value="' . esc_attr( $activation_nonce ) . '">';
	echo ' <button type="submit" class="button">Activate</button>';
	echo '</form>';

	echo '<p style="margin-top:.75rem">';
	echo '<a href="' . esc_url( 'https://pluginlicensingkit.com/app/' . rawurlencode( (string) $plugin_slug ) . '/' ) . '" target="_blank" rel="noopener">Get a License</a> | ';
	echo '<a href="https://pluginlicensingkit.com/dashboard.php" target="_blank" rel="noopener">Retrieve Your License</a>';
	echo '</p>';

	echo '<div class="plk-footer-note" style="border-top:1px solid #ccc; text-align:right; margin-top:1rem; padding-top:.5rem">';
	/* translators: 1: Plugin name, 2: opening <a> tag, 3: closing </a> tag */
	$template = __( '%1$s manages licenses through %2$sPlugin Licensing Kit%3$s.', 'sitemap-manager-pro' );
	$line     = sprintf(
		$template,
		esc_html( $plugin_name ),
		'<a href="' . esc_url( 'https://pluginlicensingkit.com/about.php' ) . '" target="_blank" rel="noopener noreferrer">',
		'</a>'
	);
	printf( '<small>%s</small>', wp_kses_post( $line ) );
	echo '</div>';

	echo '</div>';

	// VALID modal
	echo '<div id="plk-modal-valid" class="plk-modal-box plk-hidden" role="dialog" aria-modal="true" aria-labelledby="plk-valid-title" data-dismissible="1">';
	echo '<button type="button" id="plk-modal-close" class="plk-close-button" aria-label="Close">×</button>';
	echo '<h2 id="plk-valid-title">License Status</h2>';
	

	echo '<div class="plk-loader"></div>';
	echo '<div class="plk-footer-note" style="border-top:1px solid #ccc; text-align:right; margin-top:1rem; padding-top:.5rem">';
	$plk_url = 'https://pluginlicensingkit.com/app/' . rawurlencode( (string) $plugin_slug );
	/* translators: 1: Plugin name, 2: opening <a> tag, 3: closing </a> tag */
	$line = sprintf(
		__( '%1$s manages licenses through %2$sPlugin Licensing Kit%3$s.', 'sitemap-manager-pro' ),
		esc_html( $plugin_name ),
		'<a href="' . esc_url( $plk_url ) . '" target="_blank" rel="noopener">',
		'</a>'
	);
	echo '<small>' . wp_kses_post( $line ) . '</small>';
	echo '</div>';
	echo '</div>';
}

/**
 * Handle license activation submit (verify with server, then save/redirect)
 */
add_action( 'admin_init', function () {
	$method = isset( $_SERVER['REQUEST_METHOD'] ) ? strtoupper( sanitize_text_field( wp_unslash( (string) $_SERVER['REQUEST_METHOD'] ) ) ) : '';
	if ( $method !== 'POST' || ! isset( $_POST['_wpnonce'], $_POST['plk_license_key'] ) ) {
	    return;
	}
	if ( ! wp_verify_nonce( wp_unslash( $_POST['_wpnonce'] ), 'plk_activate_license' ) ) {
	    return;
	}
	if ( ! current_user_can( 'manage_options' ) ) {
	    wp_die( esc_html__( 'Sorry, you are not allowed to do that.', 'sitemap-manager-pro' ) );
	}
 
	$key = sanitize_text_field( wp_unslash( $_POST['plk_license_key'] ) );
	if ( $key === '' ) return;
 
	$result = plk_verify_license_remote( $key );
 
	if ( $result['ok'] && is_array( $result['data'] ) && ( $result['data']['status'] ?? '' ) === 'valid' ) {
	    plk_save_license_data( $key, $result['data'] );
	    set_transient( 'plk_last_license_check', true, DAY_IN_SECONDS );
	    wp_safe_redirect( admin_url( 'tools.php?page=sitemap-manager-pro' ) );
	    exit;
	}
 
	$dest = add_query_arg(
	    'plk_error',
	    $result['ok'] ? 'invalid' : 'network',
	    admin_url( 'tools.php?page=sitemap-manager-pro' )
	);
	wp_safe_redirect( $dest );
	exit;
 } );


/**
 * Clear license via POSTed form
 */
add_action( 'admin_init', function () {
	// Normalize method and posted flag for PHPCS
	$method = isset( $_SERVER['REQUEST_METHOD'] )
	    ? sanitize_text_field( wp_unslash( (string) $_SERVER['REQUEST_METHOD'] ) )
	    : '';
 
	$clear_flag = isset( $_POST['plk_clear_license'] )
	    ? sanitize_text_field( wp_unslash( $_POST['plk_clear_license'] ) )
	    : '';
 
	if ( 'POST' !== strtoupper( $method ) || '' === $clear_flag ) {
	    return;
	}
 
	check_admin_referer( 'plk_clear_license' );
 
	if ( ! current_user_can( 'manage_options' ) ) {
	    wp_die( esc_html__( 'Sorry, you are not allowed to do that.', 'sitemap-manager-pro' ) );
	}
 
	plk_clear_license_data();
 
	wp_safe_redirect( admin_url( 'tools.php?page=sitemap-manager-pro' ) );
	exit;
 } );



/**
 * Silent daily revalidation of stored license
 */
add_action( 'admin_init', function () {
	if ( get_transient( 'plk_last_license_check' ) ) {
		return;
	}
	$key = get_option( 'plk_license_key' );
	if ( ! $key ) {
		return;
	}

	$result = plk_verify_license_remote( $key );
	if ( $result['ok'] && is_array( $result['data'] ) && ( $result['data']['status'] === 'valid' ) ) {
		plk_save_license_data( $key, $result['data'] );
	} else {
		update_option( 'plk_license_status', 'expired' );
	}
	set_transient( 'plk_last_license_check', true, DAY_IN_SECONDS );
} );

	/* code for checking for updates */
	if ( ! class_exists('PLK_Updater') ) {
    class PLK_Updater {
	   private $slug;
	   private $plugin_file;
	   private $license_option;
	   private $update_check_url;
	   private $version;
	   private $param_map;

	   public function __construct(array $cfg) {
		  $this->slug             = $cfg['slug'];
		  $this->plugin_file      = $cfg['plugin_file'];
		  $this->license_option   = $cfg['license_option'] ?? 'plk_license_key';
		  $this->update_check_url = $cfg['update_check_url'];
		  $this->version          = $cfg['version'];
		  $this->param_map        = $cfg['param_map'] ?? [
			 'license' => 'license',
			 'site'    => 'site',
			 'slug'    => 'slug',
			 'version' => 'version',
		  ];

		  // Core update filters
		  add_filter('pre_set_site_transient_update_plugins', [$this, 'maybe_inject_update']);
		  add_filter('plugins_api', [$this, 'plugins_api_info'], 10, 3);

		  // Allow projects to tweak payload/response
		  // - plk/update_payload : ($payload, $slug)
		  // - plk/update_response: ($meta, $slug)
	   }

	   private function map($key) {
		  return $this->param_map[$key] ?? null; // null => don't include that field
	   }


	   // Inside PLK_Updater class…
	   
	   /** Return the site's bare domain (no scheme, no path, no "www.") */
	   private function get_site_domain(): string {
		  // Get host from home_url()
		  $host = wp_parse_url( home_url(), PHP_URL_HOST );
		  if ( ! is_string($host) || $host === '' ) {
			 // Fallbacks, just in case
			 $host = wp_parse_url( site_url(), PHP_URL_HOST );
			 if ( ! is_string($host) || $host === '' ) {
				return ''; // last resort: empty
			 }
		  }
		  $host = strtolower($host);
		  // Strip leading "www."
		  $host = preg_replace('/^www\./i', '', $host);
		  /**
		   * Allow projects to override the domain if needed (e.g., force apex).
		   * Usage: add_filter('plk/site_domain', fn($d) => 'example.com');
		   */
		  return apply_filters('plk/site_domain', $host, home_url());
	   }
	   
	   private function build_payload(): array {
		  $payload = [];
	   
		  // Only include a field if it’s mapped in param_map
		  if ($k = $this->map('license')) {
			 $payload[$k] = get_option($this->license_option, '');
		  }
		  if ($k = $this->map('site')) {
			 $payload[$k] = $this->get_site_domain(); // bare domain, no scheme/www
		  }
		  if ($k = $this->map('slug')) {
			 $payload[$k] = $this->slug;
		  }
		  if ($k = $this->map('version')) {
			 $payload[$k] = $this->version;
		  }
	   
		  // Allow overrides/injection
		  return apply_filters('plk/update_payload', $payload, $this->slug);
	   }



	   /** Call the update endpoint and parse JSON */
	   private function fetch_update_meta(): ?array {
		  if ( empty($this->update_check_url) ) return null;

		  $res = wp_remote_post($this->update_check_url, [
			 'timeout' => 10,
			 'sslverify' => true,
			 'headers' => ['Accept' => 'application/json'],
			 'body'    => $this->build_payload(),
		  ]);

		  if ( is_wp_error($res) ) return null;
		  $code = (int) wp_remote_retrieve_response_code($res);
		  if ( $code !== 200 ) return null;

		  $body = wp_remote_retrieve_body($res);
		  $json = json_decode($body, true);
		  if ( ! is_array($json) ) return null;

		  // Allow projects to adjust/validate response
		  $json = apply_filters('plk/update_response', $json, $this->slug);
		  return $json;
	   }

	   /** Inject an available update into WP's transient if server has newer version */
	   public function maybe_inject_update($transient) {
		  if ( empty($transient->checked) ) return $transient;

		  $meta = $this->fetch_update_meta();
		  if ( ! $meta || empty($meta['version']) ) return $transient;

		  $current = $this->version;
		  if ( version_compare($meta['version'], $current, '>') && ! empty($meta['download_url']) ) {
			 $plugin_basename   = plugin_basename($this->plugin_file);
			 $update            = new stdClass();
			 $update->slug      = $this->slug;
			 $update->plugin    = $plugin_basename;
			 $update->new_version = $meta['version'];
			 $update->package   = $meta['download_url'];       // zip url
			 $update->tested    = $meta['tested']    ?? '';
			 $update->requires  = $meta['requires']  ?? '';
			 $update->icons     = $meta['icons']     ?? [];     // optional
			 $update->banners   = $meta['banners']   ?? [];     // optional
			 $transient->response[$plugin_basename] = $update;
		  }

		  return $transient;
	   }

	   /** Populate the "View details" modal (plugins_api: plugin_information) */
	   public function plugins_api_info($res, $action, $args) {
		  if ( $action !== 'plugin_information' || empty($args->slug) || $args->slug !== $this->slug ) {
			 return $res;
		  }

		  $meta = $this->fetch_update_meta();
		  if ( ! $meta ) return $res;

		  $info = new stdClass();
		  $info->name         = $meta['name']         ?? $this->slug;
		  $info->slug         = $this->slug;
		  $info->version      = $meta['version']      ?? $this->version;
		  $info->last_updated = $meta['last_updated'] ?? '';
		  $info->requires     = $meta['requires']     ?? '';
		  $info->tested       = $meta['tested']       ?? '';
		  $info->sections     = $meta['sections']     ?? ['description' => ''];
		  $info->download_link= $meta['download_url'] ?? '';
		  $info->banners      = $meta['banners']      ?? [];
		  $info->icons        = $meta['icons']        ?? [];
		  return $info;
	   }
    }
}
add_action('admin_post_plk_force_update', function () {
    if ( ! current_user_can('update_plugins') ) {
	   wp_die( esc_html__('Sorry, you are not allowed to do that.', 'sitemap-manager-pro') );
    }
    check_admin_referer('plk_activate_license'); // reuse existing nonce

    if ( ! function_exists('plugin_basename') ) {
	   require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    // Force WP to re-check plugins
    delete_site_transient('update_plugins');
    if ( function_exists('wp_update_plugins') ) {
	   wp_update_plugins();
    }

    // Figure out this plugin's basename
    $plugin_meta = get_option('plk_plugin_meta', []);
    $basename    = ! empty($plugin_meta['plugin_file']) ? plugin_basename($plugin_meta['plugin_file']) : '';

    // Look for an update for THIS plugin in the transient
    $has_update   = '0';
    $update_ver   = '';
    $updates      = get_site_transient('update_plugins');

    if ( $basename && is_object($updates) && isset($updates->response[$basename]) ) {
	   $has_update = '1';
	   $update_ver = isset($updates->response[$basename]->new_version) ? (string) $updates->response[$basename]->new_version : '';
    }

    // Redirect back to the tools page, auto-open the VALID modal
    $dest = add_query_arg([
	   'page'           => 'sitemap-manager-pro',
	   'update_checked' => $has_update,               // "1" or "0"
	   'update_version' => $update_ver,               // optional
	   'force_modal'    => '1',                       // your JS opens the VALID modal on this
    ], admin_url('tools.php'));

    wp_safe_redirect($dest);
    exit;
});




