<?php
/**
 * Plugin Name:       Sitemap Manager Pro
 * Plugin URI:        https://www.96monkeys.com/wordpress-plugins/sitemap-manager/
 * Description:       See your XML sitemaps like never before. Slice 'em, dice 'em, exclude any URL, add pages outside WordPress and more for total sitemap control.
 * Version:           1.0.6
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Author:            96 Monkeys, @michaelgrover
 * Author URI:        https://96monkeys.com
 * Text Domain:       sitemap-manager-pro
 * Update URI:        https://96monkeys.com/sitemap-manager-pro
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 */
// sitemap-manager-pro/sitemap-manager-pro.php
 if ( ! defined('ABSPATH') ) exit;
 

/** Identify Pro immediately (some code may branch on this) */
if ( ! defined('SITEMAP_MANAGER_PRO') ) {
    define('SITEMAP_MANAGER_PRO', true);
}

/** Global guard: if Basic is active, auto-deactivate it (if allowed) and bail for ALL contexts */
if ( ! function_exists('is_plugin_active') || ! function_exists('deactivate_plugins') ) {
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
}

$basic_slug   = 'sitemap-manager/sitemap-manager.php';
$basic_active = function_exists('is_plugin_active') && (
    is_plugin_active( $basic_slug ) ||
    ( function_exists('is_plugin_active_for_network') && is_multisite() && is_plugin_active_for_network( $basic_slug ) )
);

if ( $basic_active ) {
    if ( current_user_can('activate_plugins') ) {
        deactivate_plugins( $basic_slug, /*silent*/ true, /*network_wide*/ is_multisite() );
        add_action('admin_notices', function () {
            echo '<div class="notice notice-warning"><p>'
               . esc_html__( 'Sitemap Manager Basic was deactivated because Sitemap Manager Pro is active (they cannot run together).', 'sitemap-manager-pro' )
               . '</p></div>';
        });
    }
    return; // stop loading Pro on this request to avoid collisions
}

/** Activation-time guard: if Basic is active during Pro activation, turn Basic off and show a one-time notice */
register_activation_hook( __FILE__, function () use ( $basic_slug ) {
    if ( ! function_exists('is_plugin_active') ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    if ( is_plugin_active( $basic_slug ) ||
         ( function_exists('is_plugin_active_for_network') && is_multisite() && is_plugin_active_for_network( $basic_slug ) ) ) {
        deactivate_plugins( $basic_slug, /*silent*/ true, /*network_wide*/ is_multisite() );
        set_transient( 'ysmap_notice_basic_deactivated', 1, MINUTE_IN_SECONDS );
    }
});

add_action('admin_notices', function () {
    if ( get_transient('ysmap_notice_basic_deactivated') ) {
        delete_transient('ysmap_notice_basic_deactivated');
        echo '<div class="notice notice-warning"><p>'
             . esc_html__( 'Sitemap Manager Basic was deactivated because Sitemap Manager Pro is active.', 'sitemap-manager-pro' )
             . '</p></div>';
    }
});

/** Core includes (safe to load now that Basic is off or not present) */
require_once plugin_dir_path(__FILE__) . 'includes/ysmap-utils.php';
require_once plugin_dir_path(__FILE__) . 'includes/export.php';
include_once plugin_dir_path(__FILE__) . 'includes/ajax.php';

/** Licensing + updater (PLK) */
if ( file_exists( __DIR__ . '/plk/plk-loader.php' ) ) {
    include_once __DIR__ . '/plk/plk-loader.php';

    plk_register_plugin([
        'slug'              => 'sitemap-manager-pro',
        'plugin_file'       => __FILE__,
        'version'           => '1.0.6',
        'license_check_url' => 'https://pluginlicensingkit.com/api/check-license.php',
        'update_check_url'  => 'https://pluginlicensingkit.com/api/check-update.php',
        'license_option'    => 'ysmap_license_key',

        // Map to the exact variable names your endpoint expects:
        'param_map'         => [
            'license' => 'license_key',
            'site'    => 'site_url',
            'slug'    => 'product_slug',
            'version' => 'installed_version',
        ],
    ]);
}


add_action('admin_post_ysmap_export_csv', 'ysmap_handle_export_csv');

add_filter('redirect_canonical', function ($redirect_url, $requested_url) {
    return $redirect_url;
}, 10, 2);

// Optional debug notice
// add_action('admin_notices', function () {
    // if (defined('WP_DEBUG') && WP_DEBUG && is_user_logged_in() && current_user_can('manage_options')) {
    //     echo '<div class="notice notice-info"><p>' . esc_html__('Sitemap Manager debug handler is active and watching for PHP 8.1 deprecation warnings.', 'sitemap-manager-pro') . '</p></div>';
    // }
// });

// Set a handler for suppressed warnings
// add_action('init', function () {
    // if (defined('WP_DEBUG') && WP_DEBUG && is_user_logged_in() && current_user_can('manage_options')) {
    //     set_error_handler(function ($errno, $errstr, $errfile, $errline) {
    //         if (
    //             (str_contains($errstr, 'Passing null to parameter') ||
    //             str_contains($errstr, 'strpos()') ||
    //             str_contains($errstr, 'str_replace()')) &&
    //             str_contains($errfile, 'wp-includes/functions.php')
    //         ) {
    //             error_log("Suppressed deprecated warning: $errstr in $errfile on line $errline");
    //             return true;
    //         }
    //         return false;
    //     }, E_DEPRECATED);
    // }
// }, 99);

add_action('plugins_loaded', function () {



    include_once plugin_dir_path(__FILE__) . 'includes/helpers.php';
    include_once plugin_dir_path(__FILE__) . 'includes/filter-excluded-posts.php';
    


    if (is_admin()) {
        add_action('admin_menu', function () {
            add_management_page(
                'Sitemap Manager Pro',
                'Sitemap Manager Pro',
                'manage_options',
                'sitemap-manager-pro',
                'sitemap_manager_admin_page_pro'
            );
        }, 99);

        function sitemap_manager_admin_page_pro() {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only view param, no data mutation
            $tab = isset($_GET['tab']) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'sitemap';

            ?>
            <h1><?php esc_html_e('Sitemap Manager Pro from 96 Monkeys', 'sitemap-manager-pro'); ?></h1>
            <div class="wrap">
                <?php include_once plugin_dir_path(__FILE__) . 'includes/tabbed_menu.php'; ?>
                <div class="mt-4">
                    <?php
                    switch ($tab) {
                        case 'add':
                            include __DIR__ . '/views/add-url.php';
                            break;
                        case 'settings':
                            include __DIR__ . '/views/settings.php';
                            break;
                        case 'llms':
                            include __DIR__ . '/views/llms-generate.php';
                        break;
                        case 'instructions':
                            include __DIR__ . '/views/instructions.php';
                            break;
                        case 'sitemap':
                        default:
                            include __DIR__ . '/views/sitemap.php';
                            break;
                    }
                    ?>
                </div>
            </div>
            <?php
        }

        add_action('admin_enqueue_scripts', function ($hook) {
            if ($hook !== 'tools_page_sitemap-manager-pro') {
                return;
            }
        
            wp_localize_script('sitemap-manager-admin', 'ysmapAjax', [
                'nonce' => wp_create_nonce('ysmap_nonce'),
            ]);
        
            wp_enqueue_script(
                'sitemap-manager-admin',
                plugin_dir_url(__FILE__) . 'assets/admin.js',
                [],
                '1.0',
                true
            );
        
            wp_enqueue_style(
                'sitemap-manager-admin-css',
                plugin_dir_url(__FILE__) . 'assets/admin.css',
                [],
                '1.0'
            );
        }, 99);

        add_filter('set-screen-option', function ($status, $option, $value) {
            return ($option === 'ysmap_per_page') ? (int) $value : $status;
        }, 10, 3);

        add_action('load-tools_page_sitemap-manager-pro', function () {
            add_screen_option('per_page', [
                'label'   => esc_html__('Items per page', 'sitemap-manager-pro'),
                'default' => 100,
                'option'  => 'ysmap_per_page',
            ]);
        }, 99);
    }

    // Detect active SEO plugin
    $active_seo = ysmap_detect_active_seo_plugin();


    if ($active_seo === 'yoast') {
        add_action('wpseo_loaded', function () {
            add_filter('wpseo_sitemap_index', function ($sitemap_index) {
                $custom_sitemap_url = home_url('/more-pages-sitemap.xml');
    
                if (strpos($sitemap_index, $custom_sitemap_url) === false) {
                    $entry  = "\t<sitemap>\n";
                    $entry .= "\t\t<loc>" . esc_url($custom_sitemap_url) . "</loc>\n";
                    $entry .= "\t\t<lastmod>" . esc_html(gmdate('c')) . "</lastmod>\n";
                    $entry .= "\t</sitemap>\n";
    
                    $sitemap_index = str_replace('</sitemapindex>', $entry . '</sitemapindex>', $sitemap_index);
                }
    
                return $sitemap_index;
            });
        });
    }



    if ($active_seo === 'rank_math') {
        add_filter('rank_math/sitemap/index', function ($sitemaps) {
            $sitemaps['more-pages'] = [
                'loc'     => home_url('/more-pages-sitemap.xml'),
                'lastmod' => gmdate('Y-m-d\TH:i:s+00:00'),
            ];
            return $sitemaps;
        });
    }

    if ($active_seo === 'aioseo') {
        add_filter('aioseo_sitemap_index', function ($output) {
            $custom_url = home_url('/more-pages-sitemap.xml');
            $lastmod = gmdate('Y-m-d\TH:i:s+00:00');

            $output .= "<sitemap>\n";
            $output .= "  <loc>{$custom_url}</loc>\n";
            $output .= "  <lastmod>{$lastmod}</lastmod>\n";
            $output .= "</sitemap>\n";

            return $output;
        });
    }

}, 99);

function ysmap_detect_active_seo_plugin() {
    if (defined('WPSEO_VERSION')) {
        return 'yoast';
    }

    if (defined('RANK_MATH_VERSION')) {
        return 'rank_math';
    }

    if (defined('AIOSEO_VERSION')) {
        return 'aioseo';
    }

    if (function_exists('the_block_template_skip_link')) {
        return 'core';
    }

    return 'none';
}

function sitemap_create_custom_urls_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'sitemap_custom_urls';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS `$table_name` (
        `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        `url` VARCHAR(255) NOT NULL,
        `title` VARCHAR(255),
        `label` VARCHAR(255),
        `last_modified` DATE,
        `priority` DECIMAL(2,1) DEFAULT 0.5,
        `change_freq` VARCHAR(20),
        `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

register_activation_hook(__FILE__, 'ysmap_on_activation');

function ysmap_on_activation() {
    add_rewrite_rule('^more-pages-sitemap\.xml$', 'index.php?more_pages_sitemap=1', 'top');
    add_rewrite_tag('%more_pages_sitemap%', '1');

    sitemap_create_custom_urls_table();
    flush_rewrite_rules();
}


add_filter('query_vars', function ($vars) {
    $vars[] = 'more_pages_sitemap';
    return $vars;
});

add_action('template_redirect', function () {
    if (get_query_var('more_pages_sitemap')) {
        status_header(200);
        header('Content-Type: application/xml; charset=utf-8');
        include plugin_dir_path(__FILE__) . 'includes/more-pages-sitemap-output.php';
        exit;
    }
});
// add_action('admin_notices', function(){
    // if ( isset($_GET['page']) && $_GET['page'] === 'sitemap-manager-pro' ) {
    //     echo '<div class="notice notice-success"><p>Sitemap Manager Pro admin loaded.</p></div>';
    // }
// });
