<?php
if ( ! defined( 'ABSPATH' ) ) exit;

// make sure helpers are loaded
require_once plugin_dir_path( __FILE__ ) . '/../includes/helpers.php';

// require_once plugin_dir_path( __FILE__ ) . '/../includes/db-custom-urls.php';

/**
 * DELETE (guarded by nonce + caps)
 */
$method = isset( $_SERVER['REQUEST_METHOD'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : '';
if (
    'POST' === $method
    && current_user_can( 'manage_options' )
    && isset( $_POST['ysmap_delete_nonce'], $_POST['sitemap_delete_id'] )
) {
    check_admin_referer( 'ysmap_delete_custom_url', 'ysmap_delete_nonce' );

    $id = isset( $_POST['sitemap_delete_id'] ) ? absint( wp_unslash( $_POST['sitemap_delete_id'] ) ) : 0; 
    if ( $id ) {
        ysmap_delete_custom_url( $id ); // wrapper
    }

    if ( isset( $_SERVER['REQUEST_URI'] ) ) {
        wp_safe_redirect( esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) );
        exit;
    }
}

/**
 * UPDATE (guarded by nonce + caps)
 */
if (
    'POST' === $method
    && current_user_can( 'manage_options' )
    && isset( $_POST['ysmap_edit_nonce'] )
) {
    check_admin_referer( 'ysmap_edit_custom_url', 'ysmap_edit_nonce' );

    $id    = isset( $_POST['sitemap_edit_id'] ) ? absint( wp_unslash( $_POST['sitemap_edit_id'] ) ) : 0;
    $title = isset( $_POST['sitemap_title'] ) ? sanitize_text_field( wp_unslash( $_POST['sitemap_title'] ) ) : '';

    // Normalize and validate the posted URL.
    $url = '';
    if ( isset( $_POST['sitemap_url'] ) ) {
        $raw_url       = trim( (string) wp_unslash( $_POST['sitemap_url'] ) );
        $sanitized_url = esc_url_raw( $raw_url );
        $validated     = wp_http_validate_url( $sanitized_url );
        $url           = is_string( $validated ) ? $validated : '';
    }

    $last_modified = isset( $_POST['sitemap_last_modified'] ) ? sanitize_text_field( wp_unslash( $_POST['sitemap_last_modified'] ) ) : '';

    // Fallback to today if the posted date isn't YYYY-MM-DD.
    if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $last_modified ) ) {
        $last_modified = gmdate( 'Y-m-d' );
    }

    if ( $id && $url ) {
        $site_host  = ysmap_normalize_host( wp_parse_url( home_url(), PHP_URL_HOST ) );
        $input_host = ysmap_normalize_host( wp_parse_url( $url, PHP_URL_HOST ) );

        // Only update if URL is on this site (www/non-www handled by the normalizer).
        if ( $input_host === $site_host ) {

            // Enrich from remote like ADD does.
            $enriched_title = $title;        // keep user-provided title unless empty
            $enriched_last  = $last_modified;

            $response = wp_remote_get( $url, array( 'timeout' => 5 ) );
            if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
                $body = wp_remote_retrieve_body( $response );

                if ( is_string( $body ) && preg_match('/<title>(.*?)<\/title>/is', $body, $m) ) {

                    $enriched_title = sanitize_text_field( $m[1] );
                }

                $lm_header = wp_remote_retrieve_header( $response, 'last-modified' );
                if ( $lm_header ) {
                    $ts = strtotime( $lm_header );
                    if ( false !== $ts ) {
                        $enriched_last = gmdate( 'Y-m-d', $ts );
                    }
                }
            }

            // Final fallbacks
            if ( $enriched_title === '' ) {
                $enriched_title = $url; // match ADD behavior
            }
            if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $enriched_last ) ) {
                $enriched_last = gmdate( 'Y-m-d' );
            }

            ysmap_update_custom_url( $id, $enriched_title, $url, $enriched_last ); // wrapper also busts cache
        }
    }
}


/**
 * ADD (guarded by nonce + caps)
 */
if ( 'POST' === $method && current_user_can( 'manage_options' ) && isset( $_POST['ysmap_add_nonce'] ) ) {
    check_admin_referer( 'ysmap_add_custom_url', 'ysmap_add_nonce' );

    // Normalize the posted custom URLs (PHPCS-safe).
    $urls_raw = '';
    if ( isset( $_POST['sitemap_custom_url'] ) ) {
        $urls_raw = sanitize_textarea_field( wp_unslash( (string) $_POST['sitemap_custom_url'] ) );
    }

    // Normalize the site host once (handles www vs non-www, case, trailing dot).
    $site_host = ysmap_normalize_host( wp_parse_url( home_url(), PHP_URL_HOST ) );

    // Split on commas or whitespace
    $url_list = array_filter( array_map( 'trim', preg_split( '/[\s,]+/', (string) $urls_raw ) ) );
    if ( $site_host && ! empty( $url_list ) ) {

        // Fetch existing once (cached), then check in-memory to avoid SELECT per URL
        $existing_rows  = ysmap_get_custom_urls(); // wrapper (cached)

        // Use a normalized key for dedupe (lowercased, no trailing slash).
        $existing_index = array();
        foreach ( $existing_rows as $row ) {
            $key = strtolower( untrailingslashit( (string) $row->url ) );
            $existing_index[ $key ] = true;
        }

        foreach ( $url_list as $url ) {
            // Sanitize, then validate as a proper http(s) URL.
            $url = esc_url_raw( $url );
            if ( empty( $url ) ) {
                continue;
            }
            $validated = wp_http_validate_url( $url );
            if ( ! $validated ) {
                continue;
            }
            $url = $validated;

            // Host check using the same normalizer (so www vs non-www match).
            $input_host = ysmap_normalize_host( wp_parse_url( $url, PHP_URL_HOST ) );
            if ( $input_host !== $site_host ) {
                continue; // external or invalid
            }

            // Dedupe against existing (normalized)
            $dupe_key = strtolower( untrailingslashit( $url ) );
            if ( isset( $existing_index[ $dupe_key ] ) ) {
                continue; // already present
            }

            // Try to enrich title/lastmod
            $title         = $url;
            $last_modified = gmdate( 'Y-m-d' );

            $response = wp_remote_get( $url, array( 'timeout' => 5 ) );
            if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
                $body = wp_remote_retrieve_body( $response );
                if ( is_string( $body ) && preg_match( '/<title>(.*?)<\/title>/is', $body, $m ) ) {
                    $title = sanitize_text_field( $m[1] );
                }
                $lm_header = wp_remote_retrieve_header( $response, 'last-modified' );
                if ( $lm_header ) {
                    $ts = strtotime( $lm_header );
                    if ( false !== $ts ) {
                        $last_modified = gmdate( 'Y-m-d', $ts );
                    }
                }
            }

            ysmap_insert_custom_url( $url, $title, $last_modified ); // wrapper; also clears cache
            $existing_index[ $dupe_key ] = true; // keep set in-memory
        }
    }
}

?>

<div class="wrap">
    <h1><?php echo esc_html__( 'Add to Sitemap', 'sitemap-manager-pro' ); ?></h1>
    <p>
        <?php
        /* translators: %s: example sitemap URL. */
        printf(
            esc_html__( 'Do you have URLs that are outside of the WordPress structure that you want to include in your sitemap? Enter a full URL here to add those pages to a new sitemap called more-pages-sitemap.xml. Important: when you submit sitemaps to search engines, also add %s', 'sitemap-manager-pro' ),
            '<code>' . esc_html( trailingslashit( home_url() ) . 'more-pages-sitemap.xml' ) . '</code>'
        );
        ?>
    </p>

    <form method="post">
        <?php wp_nonce_field( 'ysmap_add_custom_url', 'ysmap_add_nonce' ); ?>
        <label for="sitemap_custom_url">
            <?php echo esc_html__( 'Add URL(s):', 'sitemap-manager-pro' ); ?><br>
            <small>
                <?php
                /* translators: %s: site home URL */
                printf(
                    esc_html__( 'Enter one or more full URLs, separated by commas. Each must begin with "%s".', 'sitemap-manager-pro' ),
                    esc_html( home_url() )
                );
                ?>
            </small>
        </label><br>
        <textarea name="sitemap_custom_url" id="sitemap_custom_url" rows="3" style="width:100%;max-width:600px;" placeholder="<?php echo esc_attr( trailingslashit( home_url() ) . 'page-one, ' . trailingslashit( home_url() ) . 'page-two' ); ?>"></textarea>
        <br><br>
        <button type="submit" class="button button-primary"><?php echo esc_html__( 'Add URL(s)', 'sitemap-manager-pro' ); ?></button>
    </form>

    <hr>

    <h2><?php echo esc_html__( 'Added URLs', 'sitemap-manager-pro' ); ?></h2>
    <table class="widefat fixed striped">
        <thead>
            <tr>
                <th style="width:30%;"><?php echo esc_html__( 'Title', 'sitemap-manager-pro' ); ?></th>
                <th style="width:40%;"><?php echo esc_html__( 'URL', 'sitemap-manager-pro' ); ?></th>
                <th style="width:15%;"><?php echo esc_html__( 'Last Modified', 'sitemap-manager-pro' ); ?></th>
                <th style="width:15%;"><?php echo esc_html__( 'Actions', 'sitemap-manager-pro' ); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php
            $results = ysmap_get_custom_urls(); // wrapper (cached)

            if ( ! empty( $results ) ) :
                foreach ( $results as $row ) : ?>
                    <tr>
                        <td>
                            <form method="post">
                                <?php wp_nonce_field( 'ysmap_edit_custom_url', 'ysmap_edit_nonce' ); ?>
                                <input type="hidden" name="sitemap_title" value="<?php echo esc_attr( $row->display_title ); ?>">
                                <?php echo esc_html( $row->title ); ?>
                        </td>
                        <td>
                                <input type="text" name="sitemap_url" value="<?php echo esc_attr( $row->url ); ?>" style="width:100%;">
                        </td>
                        <td>
                                <input type="date" name="sitemap_last_modified" value="<?php echo esc_attr( $row->last_modified ); ?>">
                        </td>
                        <td>
                                <input type="hidden" name="sitemap_edit_id" value="<?php echo esc_attr( $row->id ); ?>">
                                <button type="submit" class="button"><?php echo esc_html__( 'Update', 'sitemap-manager-pro' ); ?></button>
                            </form>

                            <form method="post" style="display:inline;margin-left:5px;">
                                <?php wp_nonce_field( 'ysmap_delete_custom_url', 'ysmap_delete_nonce' ); ?>
                                <input type="hidden" name="sitemap_delete_id" value="<?php echo esc_attr( $row->id ); ?>">
                                <button type="submit" class="button" onclick="return confirm('<?php echo esc_attr__( 'This will delete this URL from the sitemap but will not impact the actual page. Proceed?', 'sitemap-manager-pro' ); ?>')">
                                    <?php echo esc_html__( 'Delete', 'sitemap-manager-pro' ); ?>
                                </button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach;
            else : ?>
                <tr><td colspan="4"><?php echo esc_html__( 'No custom URLs added yet.', 'sitemap-manager-pro' ); ?></td></tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<?php require_once plugin_dir_path( __DIR__ ) . '/includes/footer.php'; ?>
