<?php
if ( ! defined( 'ABSPATH' ) ) exit;

include_once plugin_dir_path(__FILE__) . '/../includes/tabbed_menu.php';
require_once plugin_dir_path(__FILE__) . '/../includes/helpers.php';

/** Tiny safe wrappers if helpers didn’t load for any reason */
if ( ! function_exists('ysmap_get_h1_info') ) {
		function ysmap_get_h1_info( WP_Post $post ): array {
				return ['count'=>0,'first'=>'','status'=>'yellow','msg'=>'H1 check unavailable'];
		}
}
if ( ! function_exists('ysmap_get_meta_desc') ) {
		function ysmap_get_meta_desc( int $post_id ): array {
				return ['text'=>'','len'=>0,'status'=>'yellow','msg'=>'Meta check unavailable'];
		}
}
if ( ! function_exists('ysmap_get_canonical_info') ) {
		function ysmap_get_canonical_info( int $post_id ): array {
				return ['url'=>'','status'=>'yellow','msg'=>'Canonical check unavailable'];
		}
}
if ( ! function_exists('ysmap_is_noindex') ) {
		function ysmap_is_noindex( $post_id ){ return false; }
}
if ( ! function_exists('ysmap_get_posts_by_directory') ) {
		function ysmap_get_posts_by_directory( $dir ){ return array(); }
}

/** Permissions */
if ( ! current_user_can('edit_others_posts') ) {
		wp_die( esc_html__('You do not have sufficient permissions to access this page.', 'sitemap-manager-pro') );
}

/** License info (kept; not gating UI here) */
$status            = get_option('ysmap_license_status');
$license           = get_option('ysmap_license_key');
$valid_until       = get_option('ysmap_license_valid_until');
$has_valid_license = !empty($license) && !empty($valid_until) && (strtotime($valid_until) >= time());

/** Request vars */
$selected_post_type = isset($_GET['post_type_filter']) ? sanitize_text_field( wp_unslash( $_GET['post_type_filter'] ) ) : 'all';
$search_term        = isset($_GET['ysmap_search'])     ? sanitize_text_field( wp_unslash( $_GET['ysmap_search'] ) ) : '';
$dir_filter         = isset($_GET['dir_filter'])       ? sanitize_text_field( wp_unslash( (string) $_GET['dir_filter'] ) ) : '';

$index_filter = isset($_GET['index_filter']) ? sanitize_key($_GET['index_filter']) : 'all';
if ( ! in_array($index_filter, array('all','noindex','index'), true) ) $index_filter = 'all';

$excluded_only   = ( isset($_GET['ysmap_excluded_only']) && $_GET['ysmap_excluded_only'] === '1' );
$excluded_active = (bool) $excluded_only;

$paged    = max( 1, isset($_GET['paged']) ? intval($_GET['paged']) : 1 );
$per_page = intval( get_option('ysmap_per_page', 100) );
if ( $per_page < 1 ) $per_page = 100;

$post_types = get_post_types( array('public'=>true), 'objects' );

/** Get current excluded list */
$excluded_ids   = get_option('ysmap_excluded_post_ids', array());
$excluded_ids   = is_array($excluded_ids) ? $excluded_ids : array();
$excluded_count = count($excluded_ids);

/** Notices dismiss */
if ( isset($_GET['ysmap_dismiss_notice']) && $_GET['ysmap_dismiss_notice'] === '1' ) {
		update_user_meta( get_current_user_id(), 'ysmap_notice_dismissed', true );
}

/** Handle POST: exclusions */
if ( isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST' && check_admin_referer('ysmap_save_exclusions') ) {

		$submitted_ids = array_map('intval', $_POST['ysmap_excluded'] ?? array());
		$existing_ids  = get_option('ysmap_excluded_post_ids', array());
		$existing_ids  = is_array($existing_ids) ? $existing_ids : array();

		$visible_ids = array();
		if ( ! empty($_POST['ysmap_visible_ids']) ) {
				$raw = wp_unslash((string) $_POST['ysmap_visible_ids']);
				$raw = preg_replace('/[^0-9,]/', '', $raw);
				$visible_ids = array_map('intval', array_filter(array_map('trim', explode(',', $raw))));
		}

		// Start from existing, remove unchecked among visible, then add checked
		$updated_ids = $existing_ids;
		if ( ! empty($visible_ids) ) {
				$to_remove   = array_diff($visible_ids, $submitted_ids);
				$updated_ids = array_diff($updated_ids, $to_remove);
		}
		$updated_ids = array_values(array_unique(array_merge($updated_ids, $submitted_ids)));

		update_option('ysmap_excluded_post_ids', $updated_ids);

		// Refresh in-memory values for this request
		$excluded_ids   = $updated_ids;
		$excluded_count = count($excluded_ids);

		// Taxonomy exclusions
		$excluded_taxonomies = isset($_POST['ysmap_excluded_taxonomies'])
				? array_map( 'sanitize_key', (array) wp_unslash( $_POST['ysmap_excluded_taxonomies'] ) )
				: array();
		update_option( 'ysmap_excluded_taxonomies', $excluded_taxonomies );

		echo '<div class="notice notice-success"><p>' . esc_html__('Exclusions saved.', 'sitemap-manager-pro') . '</p></div>';

		if ( function_exists('ysmap_clear_seo_sitemap_cache') ) {
				ysmap_clear_seo_sitemap_cache();
		}
}

/** Sorting */
$valid_orderby  = array('title','date','url');
$default_sort   = get_option('ysmap_default_sort', 'date');
$default_order  = strtoupper( get_option('ysmap_default_order', 'ASC') );

$orderby = ( isset($_GET['orderby']) && in_array($_GET['orderby'], $valid_orderby, true) )
		? sanitize_key($_GET['orderby'])
		: $default_sort;

if ( isset($_GET['order']) ) {
		$order_raw = strtolower( sanitize_text_field( wp_unslash( $_GET['order'] ) ) );
		$order     = in_array( $order_raw, array('asc','desc'), true ) ? strtoupper($order_raw) : $default_order;
} else {
		$order = $default_order;
}

/** Build main query args */
$args = array(
		'post_status'    => 'publish',
		'posts_per_page' => $per_page,
		'paged'          => $paged,
		'orderby'        => ($orderby === 'url') ? 'name' : $orderby,
		'order'          => $order,
);

/** Directory filter */
$matching_ids = ysmap_get_posts_by_directory( $dir_filter );
if ( ! is_array($matching_ids) ) $matching_ids = array();
if ($dir_filter !== '' && is_array($matching_ids)) {
		$args['post__in'] = !empty($matching_ids) ? $matching_ids : array(0);
}

/** Excluded-only */
if ( $excluded_only ) {
		if ( ! empty($excluded_ids) ) {
				$args['post__in'] = ! empty($args['post__in'])
						? array_values( array_intersect( (array) $args['post__in'], (array) $excluded_ids ) )
						: (array) $excluded_ids;
		} else {
				$args['post__in'] = array(0);
		}
}

/** Post type filter */
if ( $selected_post_type && $selected_post_type !== 'all' && $selected_post_type !== '__excluded__' ) {
		$args['post_type'] = $selected_post_type;
} else {
		$args['post_type'] = array_keys( $post_types );
}

/** Search */
if ( $search_term !== '' ) {
		$args['s'] = $search_term;
		add_filter('posts_search', function($search, $wp_query) use ($search_term){
				global $wpdb;
				$like = '%' . $wpdb->esc_like($search_term) . '%';
				return $wpdb->prepare(
						" AND ({$wpdb->posts}.post_title LIKE %s OR {$wpdb->posts}.guid LIKE %s)",
						$like, $like
				);
		}, 10, 2);
}

/** Index filter */
if ( in_array($index_filter, array('noindex','index'), true) ) {
		if ( function_exists('ysmap_apply_noindex_meta_query') ) {
				ysmap_apply_noindex_meta_query($args, $index_filter);
		}
}

/** Run queries */
$query = new WP_Query($args);

/** Count query to show filtered count in KPI */
$count_args = array(
		'post_status'    => 'publish',
		'posts_per_page' => -1,
		'fields'         => 'ids',
		'post_type'      => $args['post_type'],
);
if ( $dir_filter !== '' ) {
		if ( ! empty($matching_ids) ) {
				$count_args['post__in'] = ! empty($count_args['post__in'])
						? array_values( array_intersect( (array) $count_args['post__in'], $matching_ids ) )
						: $matching_ids;
		} else {
				$count_args['post__in'] = array(0);
		}
}
if ( $search_term !== '' ) {
		$count_args['s'] = $search_term;
		add_filter('posts_search', function($search, $wp_query) use ($search_term){
				global $wpdb;
				return $wpdb->prepare(" AND {$wpdb->posts}.post_title LIKE %s ", '%' . $wpdb->esc_like($search_term) . '%');
		}, 10, 2);
}
if ( $excluded_only ) {
		if ( ! empty($excluded_ids) ) {
				$count_args['post__in'] = ! empty($count_args['post__in'])
						? array_values( array_intersect( (array) $count_args['post__in'], (array) $excluded_ids ) )
						: (array) $excluded_ids;
		} else {
				$count_args['post__in'] = array(0);
		}
}
if ( in_array($index_filter, array('noindex','index'), true) && function_exists('ysmap_apply_noindex_meta_query') ) {
		ysmap_apply_noindex_meta_query($count_args, $index_filter);
}
$count_query = new WP_Query($count_args);
$total_found = $count_query->post_count;

/** Sorting links */
$new_title_order = ($orderby === 'title' && $order === 'ASC') ? 'desc' : 'asc';
$new_date_order  = ($orderby === 'date'  && $order === 'ASC') ? 'desc' : 'asc';
$new_url_order   = ($orderby === 'url'   && $order === 'ASC') ? 'desc' : 'asc';

$title_sort_url = esc_url( add_query_arg( array('orderby'=>'title','order'=>$new_title_order) ) );
$date_sort_url  = esc_url( add_query_arg( array('orderby'=>'date', 'order'=>$new_date_order ) ) );
$url_sort_url   = esc_url( add_query_arg( array('orderby'=>'url',  'order'=>$new_url_order  ) ) );

/** Begin output */
echo '<div class="wrap">';
echo '<h2>' . esc_html__('Sitemap Controller', 'sitemap-manager-pro') . '</h2>';
?>
<p>
	Below is your complete sitemap, enhanced with the page title for each page.
	<details style="display:inline;">
		<summary style="cursor:pointer;">See Directions</summary>
		<ol style="margin:.5em 0 0 1.25em;">
			<li><strong>Post Type</strong> Dropdown: filter sitemap by content type (Posts, Pages, CPTs).</li>
			<li><strong>Directory</strong> Dropdown: filter sitemap to see pages by URL path (e.g., <code>/blog/</code>).</li>
			<li><strong>Search</strong>: for words or terms that appear in titles and URLs.</li>
			<li><strong>NoIndex</strong> Button: show only pages marked <code>noindex</code>.</li>
			<li><strong>Excluded</strong> Button: show only URLs you’ve excluded.</li>
			<li><strong>Exclude a URL</strong>: tick its box, then click <em>Save Changes</em> at the bottom of the page.</li>
			<li><strong>Sort</strong>: click the Title, Date, or URL table headers.</li>
			<li><strong>Export Sitemap</strong>: click the download button (top right of the table) to get a CSV.</li>
		</ol>
	</details>
</p>
<?php

/** Styles */
echo '<style>
.ysmap-badge{padding:2px 6px;border-radius:3px;font-size:11px;display:inline-block;vertical-align:middle;margin-left:.5rem}
.ysmap-badge--bad{background:#f8d7da;color:#842029}
.ysmap-toolbar{display:flex;flex-wrap:wrap;gap:.5rem;align-items:center;margin:8px 0;}
#ysmap-filter-form{display:flex;flex-wrap:wrap;gap:.5rem;align-items:center;margin:0;width:100%;}
#ysmap-filter-form .ysmap-search{flex:1 1 320px;min-width:320px;max-width:320px;}
#ysmap-filter-form select{max-width:220px}
.ysmap-actions{display:flex;gap:.5rem;align-items:center;}
.ysmap-toggles{display:flex;gap:.5rem;align-items:center;width:100%;justify-content:flex-start;margin-top:.5rem;}

.ysmap-summary{display:flex;flex-wrap:wrap;align-items:center;justify-content:space-between;background:#fff;border:1px solid #e5e5e5;border-radius:6px;padding:10px 12px;margin:12px 0;}
.ysmap-kpis{display:flex;flex-wrap:wrap;gap:12px;align-items:center;}
.ysmap-kpi{display:inline-flex;gap:6px;align-items:center;font-weight:600;}
.ysmap-kpi small{font-weight:400;opacity:.75}
.ysmap-chips{display:flex;flex-wrap:wrap;gap:6px;margin-left:10px;}
.ysmap-chip{background:#f1f5f9;border:1px solid #e2e8f0;border-radius:999px;padding:2px 8px;font-size:12px;}
.ysmap-right{display:flex;align-items:center;gap:8px;}
.ysmap-viewbar-label { font-weight: 600; margin-right: 8px; opacity: .8; }

/* Toggle button in title cell */
.ysmap-seo-cell .ysmap-seo-toggle {
		display: inline-flex;
		gap: .5rem;
		align-items: center;
		background: transparent;
		border: 0;
		padding: 0;
		font-weight: 600;
		cursor: pointer;
}
.ysmap-seo-cell .ysmap-caret { display:inline-block; transition:transform .2s ease; font-size:1.25rem; line-height:.5; }

/* Details row styling */
.ysmap-seo-row td { background:#fff; border-top:none; padding:.5rem .75rem; }
.ysmap-seo-gutter { width:20px; } /* same as checkbox col */

.ysmap-seo-details {
		margin:.25rem 0 .75rem;
		padding:.75rem .9rem;
		background:#fff;
		border:1px solid #e1e4ea;
		border-radius:6px;
}

.ysmap-seo-label { display:flex; align-items:center; gap:.5rem; margin-bottom:.15rem; font-size:12.5px; }
.ysmap-seo-block + .ysmap-seo-block { margin-top:.65rem; }

.ysmap-dot { width:.55rem; height:.55rem; border-radius:50%; display:inline-block; background:#c1c7d0; }
.ysmap-dot.good { background:#1f9d55; }  /* green */
.ysmap-dot.warn { background:#e0a800; }  /* amber */
.ysmap-dot.bad  { background:#d63638; }  /* WP red */

.ysmap-seo-text { font-size:13px; line-height:1.45; color:#1e1e1e; word-break:break-word; }
.ysmap-seo-text em { color:#6a737d; }

.ysmap-seo-note { margin-top:.35rem; font-size:12.5px; line-height:1.4; padding:.35rem .5rem; border-radius:4px; background:#f8fafc; color:#495057; }
.ysmap-seo-note.warn { border-left:3px solid #e0a800; }
.ysmap-seo-note.bad  { border-left:3px solid #d63638; }
.ysmap-caret {display: none !important;}
</style>';

/** Toolbar */
echo '<div class="ysmap-toolbar">';
echo '<form method="get" id="ysmap-filter-form">';
echo '<input type="hidden" name="page" value="sitemap-manager-pro">';
echo '<input type="hidden" name="index_filter" id="ysmap-index-filter" value="' . esc_attr($index_filter) . '">';
echo '<input type="hidden" name="ysmap_excluded_only" id="ysmap-excluded-only" value="' . ( $excluded_only ? '1' : '0' ) . '">';

echo '<select name="post_type_filter" id="post_type_filter">';
echo '<option value="all">' . esc_html__('— All Post Types —', 'sitemap-manager-pro') . '</option>';
foreach ($post_types as $slug => $pt) {
		echo '<option value="' . esc_attr($slug) . '"' . selected($slug, $selected_post_type, false) . '>' . esc_html($pt->labels->singular_name) . '</option>';
}
echo '</select> ';

$dir_options = function_exists('ysmap_get_directory_filters') ? ysmap_get_directory_filters() : array();
echo '<select name="dir_filter">';
echo '<option value="">' . esc_html__('All Directories', 'sitemap-manager-pro') . '</option>';
foreach ( (array) $dir_options as $dir ) {
		echo '<option value="' . esc_attr($dir) . '" ' . selected($dir_filter, $dir, false) . '>' . esc_html($dir) . '</option>';
}
echo '</select> ';

echo '<input type="text" name="ysmap_search" class="ysmap-search" value="' . esc_attr($search_term) . '" placeholder="' . esc_attr__('Search titles and URLs', 'sitemap-manager-pro') . '" />';

echo '<span class="ysmap-actions">';
submit_button( __('Search','sitemap-manager-pro'), 'secondary', '', false );

$filters_active = ( $search_term !== '' ) || ( $selected_post_type && $selected_post_type !== 'all' ) || ( $dir_filter !== '' ) || ( $index_filter !== 'all' ) || $excluded_only;
if ( $filters_active ) {
		echo '<a href="' . esc_url( admin_url('admin.php?page=sitemap-manager-pro') ) . '" class="button button-small ysmap-clear" title="' . esc_attr__('Clear filters','sitemap-manager-pro') . '">
						<span class="dashicons dashicons-image-rotate" style="vertical-align:middle;"></span> ' . esc_html__('Clear All','sitemap-manager-pro') .
				 '</a>';
}
echo '</span>';

echo '<div class="ysmap-toggles">';
$noindex_active  = ( $index_filter === 'noindex' );
$noindex_classes = 'button' . ( $noindex_active ? ' button-primary' : '' );
$noindex_label   = $noindex_active ? __( 'NoIndex: ON','sitemap-manager-pro' ) : __( 'NoIndex','sitemap-manager-pro' );
printf(
		'<button type="button" id="ysmap-noindex-toggle" class="%1$s" aria-pressed="%2$s" title="%3$s">%4$s</button>',
		esc_attr($noindex_classes),
		esc_attr($noindex_active ? 'true' : 'false'),
		esc_attr__( 'Show only noindex items (click again to clear)','sitemap-manager-pro' ),
		esc_html($noindex_label)
);

$disabled_flag    = empty( $excluded_ids );
$excluded_classes = 'button' . ( $excluded_active ? ' button-primary' : '' );
$excluded_label   = $excluded_active ? __( 'Excluded: ON','sitemap-manager-pro' ) : __( 'Excluded','sitemap-manager-pro' );
$excluded_title   = $disabled_flag ? __( 'No excluded URLs yet','sitemap-manager-pro' ) : __( 'Show only excluded URLs (click again to clear)','sitemap-manager-pro' );
printf(
		'<button type="button" id="ysmap-excluded-toggle" class="%1$s" aria-pressed="%2$s"%3$s title="%4$s">%5$s</button>',
		esc_attr($excluded_classes),
		esc_attr($excluded_active ? 'true' : 'false'),
		disabled( $disabled_flag, true, false ),
		esc_attr($excluded_title),
		esc_html($excluded_label)
);
echo '</div>';

echo '<script>
(function(){
	var form   = document.getElementById("ysmap-filter-form");
	var idxInp = document.getElementById("ysmap-index-filter");
	var idxBtn = document.getElementById("ysmap-noindex-toggle");
	var excBtn = document.getElementById("ysmap-excluded-toggle");
	if (!form) return;

	function resetToPage1(){
		var oldPaged = form.querySelector(\'input[name="paged"]\');
		if (oldPaged) oldPaged.remove();
		var paged = document.createElement("input");
		paged.type = "hidden"; paged.name = "paged"; paged.value = "1";
		form.appendChild(paged);
	}

	if (idxBtn && idxInp) {
		idxBtn.addEventListener("click", function(e){
			e.preventDefault();
			var turningOn = (idxInp.value !== "noindex");
			idxInp.value = turningOn ? "noindex" : "all";
			idxBtn.setAttribute("aria-pressed", turningOn ? "true" : "false");
			idxBtn.textContent = turningOn ? "NoIndex: ON" : "NoIndex";
			if (turningOn) { idxBtn.classList.add("button-primary"); } else { idxBtn.classList.remove("button-primary"); }
			resetToPage1(); idxBtn.disabled = true; form.submit();
		});
	}

	if (excBtn) {
		excBtn.addEventListener("click", function(e){
			e.preventDefault();
			var exclInp = document.getElementById("ysmap-excluded-only");
			if (!exclInp) {
				exclInp = document.createElement("input");
				exclInp.type = "hidden";
				exclInp.name = "ysmap_excluded_only";
				exclInp.id   = "ysmap-excluded-only";
				form.appendChild(exclInp);
			}
			var turningOn = (exclInp.value !== "1");
			exclInp.value = turningOn ? "1" : "0";
			excBtn.setAttribute("aria-pressed", turningOn ? "true" : "false");
			excBtn.textContent = turningOn ? "Excluded: ON" : "Excluded";
			if (turningOn) { excBtn.classList.add("button-primary"); } else { excBtn.classList.remove("button-primary"); }
			resetToPage1(); excBtn.disabled = true; form.submit();
		});
	}
})();
</script>';

echo '</form>'; // #ysmap-filter-form
echo '</div>'; // .ysmap-toolbar

/** Summary bar */
$total_url_count = function_exists('ysmap_get_total_url_count') ? ysmap_get_total_url_count() : 0;
echo '<div class="ysmap-summary" role="region" aria-labelledby="ysmap-viewbar-title">';
echo '<div class="ysmap-kpis">';
echo '<span id="ysmap-viewbar-title" class="ysmap-viewbar-label">' . esc_html__('Sitemap Viewbar', 'sitemap-manager-pro') . '</span>';
echo '<span class="ysmap-kpi">' . esc_html( number_format_i18n($total_url_count) ) . ' <small>' . esc_html__('total','sitemap-manager-pro') . '</small></span>';
echo '<span class="ysmap-kpi">' . esc_html( number_format_i18n($excluded_count) ) . ' <small>' . esc_html__('excluded','sitemap-manager-pro') . '</small></span>';

$chips = array();
if ( $excluded_only )                 $chips[] = esc_html__('Excluded only','sitemap-manager-pro');
if ( $selected_post_type && $selected_post_type !== 'all' ) {
		$label = isset($post_types[$selected_post_type]) ? $post_types[$selected_post_type]->labels->singular_name : $selected_post_type;
		$chips[] = esc_html__('Type: ','sitemap-manager-pro') . esc_html($label);
}
if ( $dir_filter !== '' )             $chips[] = esc_html__('Dir: ','sitemap-manager-pro') . esc_html($dir_filter);
if ( $search_term !== '' )            $chips[] = esc_html__('Search: ','sitemap-manager-pro') . '“' . esc_html($search_term) . '”';
if ( $index_filter === 'noindex' )    $chips[] = esc_html__('NoIndex','sitemap-manager-pro');
if ( $index_filter === 'index' )      $chips[] = esc_html__('Indexable','sitemap-manager-pro');

if ( ! empty($chips) ) {
		echo '<span class="ysmap-kpi">' . esc_html( number_format_i18n($total_found) ) . ' <small>' . esc_html__('filtered','sitemap-manager-pro') . '</small></span>';
		echo '<span class="ysmap-chips">';
		foreach ( $chips as $chip ) echo '<span class="ysmap-chip">' . esc_html($chip) . '</span>';
		echo '</span>';
}
echo '</div>';

echo '<div class="ysmap-right">';
echo '<form method="post" action="' . esc_url( admin_url('admin-post.php') ) . '" style="display:inline;">';
wp_nonce_field('ysmap_export_csv', 'ysmap_export_csv_nonce');
echo '<input type="hidden" name="action" value="ysmap_export_csv">';
echo '<button type="submit" class="button" title="' . esc_attr__('Export Entire Sitemap (.csv)', 'sitemap-manager-pro') . '">';
echo '<span class="dashicons dashicons-download" style="vertical-align: middle;"></span>';
echo '</button>';
echo '</form>';
echo '</div>';

echo '</div>'; // summary

/** TABLE + POST FORM for saving exclusions */
echo '<form method="post">';
wp_nonce_field('ysmap_save_exclusions');

echo '<table class="widefat fixed">';
echo '<thead><tr>';
echo '<th style="width:20px;"><input type="checkbox" id="ysmap_check_all" style="margin-left:0;"></th>';
echo '<th><a href="' . esc_url($title_sort_url) . '">' . esc_html__('Title','sitemap-manager-pro') . ( $orderby==='title' ? ( $order==='ASC' ? ' ▲' : ' ▼' ) : '' ) . '</a></th>';
echo '<th style="width:100px;"><a href="' . esc_url($date_sort_url)  . '">' . esc_html__('Date','sitemap-manager-pro')  . ( $orderby==='date'  ? ( $order==='ASC' ? ' ▲' : ' ▼' ) : '' ) . '</a></th>';
echo '<th style="width:100px;">' . esc_html__('Post Type','sitemap-manager-pro') . '</th>';
echo '<th><a href="' . esc_url($url_sort_url)   . '">' . esc_html__('URL','sitemap-manager-pro')   . ( $orderby==='url'   ? ( $order==='ASC' ? ' ▲' : ' ▼' ) : '' ) . '</a></th>';
echo '</tr></thead><tbody>';

$ysmap_rendered_ids = array();

if ( $query->have_posts() ) {
		while ( $query->have_posts() ) {
				$query->the_post();
				$id = get_the_ID();

				// Index filter (hard check; belt & suspenders)
				$is_noindex = ysmap_is_noindex($id);
				if ( $index_filter === 'noindex' && ! $is_noindex ) { continue; }
				if ( $index_filter === 'index'   &&   $is_noindex ) { continue; }

				$checked       = in_array( $id, $excluded_ids, true );
				$row_class_val = $checked ? 'ysmap-excluded-row' : '';

				$ysmap_rendered_ids[] = $id;

				echo '<tr' . ( $row_class_val ? ' class="' . esc_attr($row_class_val) . '"' : '' ) . '>';

				// Checkbox cell
				echo '<td><input type="checkbox" class="ysmap-post-row" name="ysmap_excluded[]" value="' . esc_attr( $id ) . '"' . checked( $checked, true, false ) . '></td>';

				// Title cell (toggle button included)
				$title      = get_the_title();
				$badge_html = $is_noindex ? ' <span class="ysmap-badge ysmap-badge--bad" title="Robots: noindex">No Index</span>' : '';
				$target_id  = 'ysmap-seo-' . $id;

				echo '<td class="ysmap-seo-cell">'
					 . esc_html( $title ) . wp_kses_post( $badge_html )
					 . ' <button type="button" class="ysmap-seo-toggle" aria-expanded="false" aria-controls="' . esc_attr($target_id) . '" data-target="' . esc_attr($target_id) . '">'
					 .     '<span class="ysmap-caret" aria-hidden="true">▸</span> '
					 .     '<span class="ysmap-seo-title"></span>'
					 .   '</button>'
					 . '</td>';

				// Date, type, URL
				echo '<td>' . esc_html( gmdate( 'Y-m-d', get_post_time( 'U', true, $id ) ) ) . '</td>';
				$type      = get_post_type( $id );
				$permalink = get_permalink( $id );
				$url_path  = wp_parse_url( $permalink, PHP_URL_PATH );
				echo '<td>' . esc_html( $type ) . '</td>';
				echo '<td><a href="' . esc_url( $permalink ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( $url_path ) . '</a></td>';

				echo '</tr>';

				/** Build the full-width SEO details row under the main row */
				global $post;

				// Fetch SEO data
				$h1_raw    = ysmap_get_h1_info( $post );
				$meta      = ysmap_get_meta_desc( $post->ID );
				$canon     = ysmap_get_canonical_info( $post->ID );
				$permalink = get_permalink( $post );

				// H1 analysis with reliable count (fallback to content scan)
				$h1_count = null; $h1_text = '';
				if (is_array($h1_raw)) {
						$h1_text  = trim($h1_raw['first'] ?? '');
						if (isset($h1_raw['count'])) $h1_count = (int)$h1_raw['count'];
				} elseif (is_string($h1_raw)) {
						$h1_text  = trim($h1_raw);
						$h1_count = ($h1_text !== '') ? 1 : 0;
				}
				if ($h1_count === null) {
						$matches  = [];
						$h1_count = preg_match_all('/<h1\b[^>]*>.*?<\/h1>/is', (string) get_post_field('post_content', $post->ID), $matches);
						if ($h1_text === '' && !empty($matches[0])) {
								$h1_text = trim( wp_strip_all_tags( $matches[0][0] ) );
						}
				}
				$h1_msg   = '';
				$h1_state = 'good';
				if ($h1_count === 0) { $h1_msg = 'No H1 found.'; $h1_state = 'bad'; }
				elseif ($h1_count > 1) { $h1_msg = 'Best practices say that a page should only have 1 H1.'; $h1_state = 'bad'; }

				// Meta
				$meta_text      = is_string($meta) ? trim($meta) : (is_array($meta) ? trim($meta['text'] ?? '') : '');
				$meta_has_token = strpos($meta_text, '%%') !== false;
				$meta_msg       = '';
				$meta_state     = 'good';
				if ($meta_text === '') { $meta_msg = 'No meta description found. Search engines may auto-generate one.'; $meta_state = 'bad'; }
				elseif ($meta_has_token) { $meta_msg = 'Template tokens detected. Confirm the rendered meta reads well.'; $meta_state = 'warn'; }

				// Canonical
				$canon_url   = '';
				$canon_state = 'none';
				if (is_array($canon) && !empty($canon['url']))      $canon_url = trim($canon['url']);
				elseif (is_string($canon) && $canon !== '')         $canon_url = trim($canon);
				if ($canon_url === '') { $canon_msg = 'No explicit canonical found.'; $canon_state = 'none'; }
				else {
						$norm = static function($u){ $u = rtrim($u, '/'); return $u === '' ? '/' : $u; };
						if ($norm($canon_url) === $norm($permalink)) { $canon_msg = 'Canonical matches page URL.'; $canon_state = 'match'; }
						else { $canon_msg = 'Canonical does not match page URL.'; $canon_state = 'mismatch'; }
				}
				$canon_state_class = ($canon_state === 'match') ? 'good' : (($canon_state === 'mismatch') ? 'bad' : 'warn');

				// Render details row
				echo '<tr class="ysmap-seo-row" id="' . esc_attr($target_id) . '" hidden>';
				echo '  <td class="ysmap-seo-gutter"></td>';
				echo '  <td colspan="4">';
				echo '    <div class="ysmap-seo-details">';

				echo '      <div class="ysmap-seo-block">';
				echo '        <div class="ysmap-seo-label"><span class="ysmap-dot ' . esc_attr($h1_state) . '"></span><strong>H1</strong>' . ( $h1_count > 0 ? ' <small>(' . (int)$h1_count . ')</small>' : '' ) . '</div>';
				echo '        <div class="ysmap-seo-text">' . ( $h1_text !== '' ? esc_html($h1_text) : '<em>No H1 found.</em>' ) . '</div>';
				if ($h1_msg) echo '  <div class="ysmap-seo-note bad">' . esc_html($h1_msg) . '</div>';
				echo '      </div>';

				echo '      <div class="ysmap-seo-block">';
				echo '        <div class="ysmap-seo-label"><span class="ysmap-dot ' . esc_attr($meta_state) . '"></span><strong>Meta Description</strong></div>';
				echo '        <div class="ysmap-seo-text">' . ( $meta_text !== '' ? esc_html($meta_text) : '<em>No meta description.</em>' ) . '</div>';
				if ($meta_msg) echo '  <div class="ysmap-seo-note ' . ($meta_state === 'bad' ? 'bad' : 'warn') . '">' . esc_html($meta_msg) . '</div>';
				echo '      </div>';

				echo '      <div class="ysmap-seo-block">';
				echo '        <div class="ysmap-seo-label"><span class="ysmap-dot ' . esc_attr($canon_state_class) . '"></span><strong>Canonical</strong></div>';
				echo '        <div class="ysmap-seo-text">' . esc_html($canon_msg) . '</div>';
				echo '      </div>';

				echo '    </div>';
				echo '  </td>';
				echo '</tr>';

		}
		wp_reset_postdata();
} else {
		echo '<tr><td colspan="5">' . esc_html__('No entries found.','sitemap-manager-pro') . '</td></tr>';
}

echo '</tbody></table>';

/** Check-all JS */
echo '<script>
(function () {
	var header = document.getElementById("ysmap_check_all");
	if (!header) return;

	function rows() { return Array.prototype.slice.call(document.querySelectorAll("input.ysmap-post-row")); }

	var baseline = new Map();
	rows().forEach(function (cb) { baseline.set(cb.value, !!cb.checked); });

	var headerMassSelected = false;

	function setHeaderState() {
		var enabled = rows().filter(function (cb) { return !cb.disabled; });
		if (enabled.length === 0) { header.checked = false; header.indeterminate = false; return; }
		var checkedCount = enabled.filter(function (cb) { return cb.checked; }).length;
		header.checked = (checkedCount === enabled.length);
		header.indeterminate = (checkedCount > 0 && checkedCount < enabled.length);
	}

	header.addEventListener("change", function () {
		if (header.checked) {
			rows().forEach(function (cb) { if (!cb.disabled) cb.checked = true; });
			headerMassSelected = true;
		} else {
			if (headerMassSelected) {
				rows().forEach(function (cb) { if (!cb.disabled) cb.checked = !!baseline.get(cb.value); });
				headerMassSelected = false;
			} else {
				rows().forEach(function (cb) { if (!cb.disabled) cb.checked = false; });
			}
		}
		setHeaderState();
	});

	document.addEventListener("change", function (e) {
		if (e.target && e.target.classList && e.target.classList.contains("ysmap-post-row")) {
			headerMassSelected = false;
			setHeaderState();
		}
	});

	setHeaderState();
})();
</script>';

/** Pagination */
$total_pages = $query->max_num_pages;
if ( $total_pages > 1 ) {
		echo '<div class="tablenav"><div class="tablenav-pages">';
		echo wp_kses_post( paginate_links( array(
				'base'    => add_query_arg( 'paged', '%#%' ),
				'format'  => '',
				'current' => $paged,
				'total'   => $total_pages,
		) ) );
		echo '</div></div>';
}

/** Taxonomy exclude options (same POST form) */
$excluded_taxonomies = get_option('ysmap_excluded_taxonomies', array());
$excluded_taxonomies = is_array($excluded_taxonomies) ? $excluded_taxonomies : array();
$taxonomies = get_taxonomies( array('public'=>true), 'objects' );

if ( get_option('ysmap_enable_taxonomy') ) {
		echo '<h2>' . esc_html__('Exclude Entire Taxonomies from Sitemap','sitemap-manager-pro') . '</h2>';
		echo '<ul>';
		foreach ( $taxonomies as $slug => $tax ) {
				if ( $slug === 'post_format' ) continue;
				$example = '/' . $slug . '/example/';
				echo '<li class="ysmap-taxonomy-row" style="display:block;">';
				echo '<label>';
				echo '<input type="checkbox" name="ysmap_excluded_taxonomies[]" value="' . esc_attr($slug) . '" data-taxlabel="' . esc_attr($tax->labels->singular_name) . '" ' . checked( in_array($slug, $excluded_taxonomies, true), true, false ) . '>';
				echo ' ' . esc_html($tax->labels->singular_name) . ' <em>ex: ' . esc_html($example) . '</em>';
				echo '</label>';
				echo '</li>';
		}
		echo '</ul>';
}

/** Hidden field of IDs that actually rendered on this page */
echo '<input type="hidden" name="ysmap_visible_ids" value="' . esc_attr( implode(',', array_map('intval', $ysmap_rendered_ids)) ) . '">';

/** Save button */
submit_button( __('Save Changes','sitemap-manager-pro') );

echo '</form>'; // end POST form
echo '</div>'; // .wrap

/** Row-based SEO details toggler */
echo "<script>
document.addEventListener('click', function(e){
	var btn = e.target.closest('.ysmap-seo-toggle');
	if (!btn) return;
	var id = btn.getAttribute('data-target') || btn.getAttribute('aria-controls');
	if (!id) return;
	var row = document.getElementById(id);
	if (!row) return;

	var caret = btn.querySelector('.ysmap-caret');
	var expanded = btn.getAttribute('aria-expanded') === 'true';
	btn.setAttribute('aria-expanded', String(!expanded));

	if (row.hasAttribute('hidden')) {
		row.removeAttribute('hidden');
		if (caret) caret.style.transform = 'rotate(90deg)';
	} else {
		row.setAttribute('hidden', '');
		if (caret) caret.style.transform = 'rotate(0deg)';
	}
});
</script>";

require_once plugin_dir_path(__DIR__) . '/includes/footer.php';
